<?php
// Hata raporlamayı etkinleştir
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// OPTIONS isteği için
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    exit(0);
}

// Debug log
error_log("dua-auth.php çağrıldı - Method: " . $_SERVER['REQUEST_METHOD'] . ", Action: " . ($_GET['action'] ?? 'none'));

try {
    require_once '../config/database.php';
} catch (Exception $e) {
    error_log("Database require hatası: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Database dosyası yüklenemedi: ' . $e->getMessage()
    ]);
    exit;
}

// Veritabanı bağlantısı (stories.php ile aynı mantık)
try {
    $pdo = getDB();
    
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
} catch(Exception $e) {
    error_log("Database connection error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Veritabanı bağlantı hatası: ' . $e->getMessage()
    ]);
    exit;
}

$action = $_GET['action'] ?? '';

switch($action) {
    case 'register':
        handleRegister($pdo);
        break;
    case 'login':
        handleLogin($pdo);
        break;
    case 'verify':
        handleVerify($pdo);
        break;
    case 'forgot_password':
        handleForgotPassword($pdo);
        break;
    case 'reset_password':
        handleResetPassword($pdo);
        break;
    case 'check_username':
        handleCheckUsername($pdo);
        break;
    case 'check_email':
        handleCheckEmail($pdo);
        break;
    default:
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Geçersiz işlem'
        ]);
        break;
}

/**
 * Kullanıcı kayıt işlemi
 */
function handleRegister($pdo) {
    error_log("handleRegister() fonksiyonu çağrıldı");
    
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        error_log("Hatalı HTTP metodu: " . $_SERVER['REQUEST_METHOD']);
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece POST metodu desteklenir']);
        return;
    }
    
    $raw_input = file_get_contents('php://input');
    error_log("Ham POST verisi: " . $raw_input);
    
    $input = json_decode($raw_input, true);
    error_log("JSON decode sonucu: " . print_r($input, true));
    
    // Gerekli alanları kontrol et
    $required_fields = ['username', 'email', 'password', 'full_name'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => ucfirst($field) . ' alanı zorunludur'
            ]);
            return;
        }
    }
    
    $username = trim($input['username']);
    $email = trim($input['email']);
    $password = $input['password'];
    $full_name = trim($input['full_name']);
    $phone = trim($input['phone'] ?? '');
    
    // Validasyonlar
    if (strlen($username) < 3 || strlen($username) > 50) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Kullanıcı adı 3-50 karakter arasında olmalıdır'
        ]);
        return;
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Geçerli bir e-posta adresi giriniz'
        ]);
        return;
    }
    
    if (strlen($password) < 6) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Şifre en az 6 karakter olmalıdır'
        ]);
        return;
    }
    
    if (strlen($full_name) < 2 || strlen($full_name) > 100) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Ad soyad 2-100 karakter arasında olmalıdır'
        ]);
        return;
    }
    
    try {
        // Kullanıcı adı kontrolü
        $stmt = $pdo->prepare("SELECT id FROM dua_users WHERE username = ?");
        $stmt->execute([$username]);
        if ($stmt->fetch()) {
            http_response_code(409);
            echo json_encode([
                'success' => false,
                'message' => 'Bu kullanıcı adı zaten kullanılıyor'
            ]);
            return;
        }
        
        // E-posta kontrolü
        $stmt = $pdo->prepare("SELECT id FROM dua_users WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetch()) {
            http_response_code(409);
            echo json_encode([
                'success' => false,
                'message' => 'Bu e-posta adresi zaten kullanılıyor'
            ]);
            return;
        }
        
        // Şifreyi hashle
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        // Doğrulama token'ı oluştur
        $verification_token = bin2hex(random_bytes(32));
        
        // Kullanıcıyı kaydet
        $stmt = $pdo->prepare("
            INSERT INTO dua_users (username, email, password, full_name, phone, verification_token) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $username,
            $email,
            $hashed_password,
            $full_name,
            $phone,
            $verification_token
        ]);
        
        $user_id = $pdo->lastInsertId();
        
        // Başarılı kayıt
        echo json_encode([
            'success' => true,
            'message' => 'Kayıt başarılı! Hesabınızı doğrulamak için e-posta adresinizi kontrol edin.',
            'data' => [
                'user_id' => $user_id,
                'username' => $username,
                'email' => $email,
                'full_name' => $full_name,
                'verification_token' => $verification_token,
                'is_verified' => false
            ]
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Kayıt sırasında bir hata oluştu'
        ]);
    }
}

/**
 * Kullanıcı giriş işlemi
 */
function handleLogin($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece POST metodu desteklenir']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['username']) || empty($input['password'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Kullanıcı adı ve şifre gereklidir'
        ]);
        return;
    }
    
    $username = trim($input['username']);
    $password = $input['password'];
    
    try {
        // Kullanıcıyı bul (username veya email ile)
        $stmt = $pdo->prepare("
            SELECT id, username, email, password, full_name, phone, is_active, is_verified 
            FROM dua_users 
            WHERE (username = ? OR email = ?) AND is_active = 1
        ");
        $stmt->execute([$username, $username]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Kullanıcı adı veya şifre hatalı'
            ]);
            return;
        }
        
        // Şifre kontrolü
        if (!password_verify($password, $user['password'])) {
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Kullanıcı adı veya şifre hatalı'
            ]);
            return;
        }
        
        // E-posta doğrulama kontrolü kaldırıldı - doğrudan giriş yapabilir
        // Artık is_verified kontrolü yapılmıyor
        
        // Son giriş zamanını güncelle
        $stmt = $pdo->prepare("UPDATE dua_users SET last_login = NOW() WHERE id = ?");
        $stmt->execute([$user['id']]);
        
        // Başarılı giriş
        unset($user['password']); // Şifreyi response'dan çıkar
        
        echo json_encode([
            'success' => true,
            'message' => 'Giriş başarılı',
            'data' => [
                'user' => $user,
                'login_time' => date('Y-m-d H:i:s')
            ]
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Giriş sırasında bir hata oluştu'
        ]);
    }
}

/**
 * E-posta doğrulama
 */
function handleVerify($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece POST metodu desteklenir']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['token'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Doğrulama token\'ı gereklidir'
        ]);
        return;
    }
    
    $token = trim($input['token']);
    
    try {
        $stmt = $pdo->prepare("
            SELECT id, username, email, is_verified 
            FROM dua_users 
            WHERE verification_token = ? AND is_active = 1
        ");
        $stmt->execute([$token]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'message' => 'Geçersiz doğrulama token\'ı'
            ]);
            return;
        }
        
        if ($user['is_verified']) {
            echo json_encode([
                'success' => true,
                'message' => 'Hesabınız zaten doğrulanmış',
                'data' => ['user_id' => $user['id']]
            ]);
            return;
        }
        
        // Hesabı doğrula
        $stmt = $pdo->prepare("
            UPDATE dua_users 
            SET is_verified = 1, verification_token = NULL 
            WHERE id = ?
        ");
        $stmt->execute([$user['id']]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Hesabınız başarıyla doğrulandı',
            'data' => ['user_id' => $user['id']]
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Doğrulama sırasında bir hata oluştu'
        ]);
    }
}

/**
 * Kullanıcı adı kontrolü
 */
function handleCheckUsername($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece GET metodu desteklenir']);
        return;
    }
    
    $username = trim($_GET['username'] ?? '');
    
    if (empty($username)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Kullanıcı adı gereklidir'
        ]);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT id FROM dua_users WHERE username = ?");
        $stmt->execute([$username]);
        $exists = $stmt->fetch() ? true : false;
        
        echo json_encode([
            'success' => true,
            'available' => !$exists,
            'message' => $exists ? 'Bu kullanıcı adı kullanılıyor' : 'Kullanıcı adı uygun'
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Kontrol sırasında bir hata oluştu'
        ]);
    }
}

/**
 * E-posta kontrolü
 */
function handleCheckEmail($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece GET metodu desteklenir']);
        return;
    }
    
    $email = trim($_GET['email'] ?? '');
    
    if (empty($email)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'E-posta adresi gereklidir'
        ]);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT id FROM dua_users WHERE email = ?");
        $stmt->execute([$email]);
        $exists = $stmt->fetch() ? true : false;
        
        echo json_encode([
            'success' => true,
            'available' => !$exists,
            'message' => $exists ? 'Bu e-posta adresi kullanılıyor' : 'E-posta adresi uygun'
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Kontrol sırasında bir hata oluştu'
        ]);
    }
}

/**
 * Şifre sıfırlama isteği
 */
function handleForgotPassword($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece POST metodu desteklenir']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['email'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'E-posta adresi gereklidir'
        ]);
        return;
    }
    
    $email = trim($input['email']);
    
    try {
        $stmt = $pdo->prepare("SELECT id, username FROM dua_users WHERE email = ? AND is_active = 1");
        $stmt->execute([$email]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            // Güvenlik için her zaman başarılı mesajı döndür
            echo json_encode([
                'success' => true,
                'message' => 'Eğer bu e-posta adresi sistemde kayıtlıysa, şifre sıfırlama bağlantısı gönderildi'
            ]);
            return;
        }
        
        // Reset token oluştur
        $reset_token = bin2hex(random_bytes(32));
        
        $stmt = $pdo->prepare("
            UPDATE dua_users 
            SET verification_token = ? 
            WHERE id = ?
        ");
        $stmt->execute([$reset_token, $user['id']]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Şifre sıfırlama bağlantısı e-posta adresinize gönderildi',
            'data' => [
                'reset_token' => $reset_token // Gerçek uygulamada bu e-posta ile gönderilir
            ]
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'İşlem sırasında bir hata oluştu'
        ]);
    }
}

/**
 * Şifre sıfırlama
 */
function handleResetPassword($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece POST metodu desteklenir']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['token']) || empty($input['password'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Token ve yeni şifre gereklidir'
        ]);
        return;
    }
    
    $token = trim($input['token']);
    $password = $input['password'];
    
    if (strlen($password) < 6) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Şifre en az 6 karakter olmalıdır'
        ]);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT id FROM dua_users 
            WHERE verification_token = ? AND is_active = 1
        ");
        $stmt->execute([$token]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$user) {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'message' => 'Geçersiz veya süresi dolmuş token'
            ]);
            return;
        }
        
        // Şifreyi güncelle
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        $stmt = $pdo->prepare("
            UPDATE dua_users 
            SET password = ?, verification_token = NULL 
            WHERE id = ?
        ");
        $stmt->execute([$hashed_password, $user['id']]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Şifreniz başarıyla güncellendi'
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Şifre güncelleme sırasında bir hata oluştu'
        ]);
    }
}
?>
