<?php
/**
 * Story Görüntülenme API Endpoint
 * Namaz Vakitleri Admin Panel
 * Mobil uygulama için story görüntülenme istatistikleri
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// OPTIONS request için
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/database.php';

/**
 * JSON response gönder
 */
function sendResponse($success, $data = null, $message = '', $code = 200) {
    http_response_code($code);
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data,
        'timestamp' => date('Y-m-d H:i:s')
    ], JSON_UNESCAPED_UNICODE);
    exit();
}

/**
 * Hata response gönder
 */
function sendError($message, $code = 400) {
    sendResponse(false, null, $message, $code);
}

try {
    $db = getDB();
    
    // Request method kontrolü
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        sendError('Sadece POST metodu desteklenir', 405);
    }
    
    // JSON input al
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        sendError('Geçersiz JSON formatı');
    }
    
    // Parametreleri al ve validate et
    $story_id = isset($input['story_id']) ? (int)$input['story_id'] : 0;
    $device_type = isset($input['device_type']) ? trim($input['device_type']) : 'mobile';
    
    if ($story_id <= 0) {
        sendError('Geçerli bir story ID gereklidir');
    }
    
    // Device type kontrolü
    $allowed_device_types = ['mobile', 'tablet', 'web'];
    if (!in_array($device_type, $allowed_device_types)) {
        $device_type = 'mobile';
    }
    
    // Story'nin var olup olmadığını kontrol et
    $stmt = $db->prepare("SELECT id FROM story_cards WHERE id = ? AND is_active = 1");
    $stmt->execute([$story_id]);
    
    if (!$stmt->fetch()) {
        sendError('Story bulunamadı veya aktif değil', 404);
    }
    
    // Transaction başlat
    $db->beginTransaction();
    
    try {
        // Story'nin view_count'unu artır
        $stmt = $db->prepare("UPDATE story_cards SET view_count = view_count + 1 WHERE id = ?");
        $stmt->execute([$story_id]);
        
        // Günlük istatistik tablosuna ekle veya güncelle
        $today = date('Y-m-d');
        
        $stmt = $db->prepare("
            INSERT INTO story_stats (story_id, date_viewed, view_count, device_type) 
            VALUES (?, ?, 1, ?)
            ON DUPLICATE KEY UPDATE 
            view_count = view_count + 1,
            device_type = VALUES(device_type)
        ");
        $stmt->execute([$story_id, $today, $device_type]);
        
        // Transaction commit
        $db->commit();
        
        // Güncellenmiş view count'u al
        $stmt = $db->prepare("SELECT view_count FROM story_cards WHERE id = ?");
        $stmt->execute([$story_id]);
        $updated_story = $stmt->fetch();
        
        $response_data = [
            'story_id' => $story_id,
            'new_view_count' => (int)$updated_story['view_count'],
            'device_type' => $device_type,
            'date' => $today
        ];
        
        sendResponse(true, $response_data, 'Görüntülenme başarıyla kaydedildi');
        
    } catch (Exception $e) {
        // Transaction rollback
        $db->rollback();
        throw $e;
    }
    
} catch (PDOException $e) {
    error_log('Database Error: ' . $e->getMessage());
    sendError('Veritabanı hatası oluştu', 500);
} catch (Exception $e) {
    error_log('General Error: ' . $e->getMessage());
    sendError('Sunucu hatası oluştu', 500);
}
?>
