<?php
// Error reporting'i aç
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);
ini_set('error_log', '../logs/php_errors.log');

// Doğru dosya yolunu kullan
require_once __DIR__ . '/../config/database.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

try {
    // Veritabanı bağlantısı
    $database = new Database();
    $pdo = $database->getConnection();
    
    if (!$pdo) {
        throw new Exception("Veritabanı bağlantısı kurulamadı");
    }
    
    error_log("DUA-COMMENTS API: Veritabanı bağlantısı başarılı");
} catch (Exception $e) {
    error_log("DUA-COMMENTS API: Veritabanı bağlantı hatası: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'Database connection failed: ' . $e->getMessage()]);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

error_log("DUA-COMMENTS API: Method: $method");
error_log("DUA-COMMENTS API: Input: " . json_encode($input));

try {
    switch ($method) {
        case 'GET':
            handleGetComments();
            break;
        case 'POST':
            handleAddComment();
            break;
        case 'PUT':
            handleUpdateComment();
            break;
        case 'DELETE':
            handleDeleteComment();
            break;
        default:
            http_response_code(405);
            echo json_encode(['error' => 'Method not allowed']);
            break;
    }
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error: ' . $e->getMessage()]);
}

function handleGetComments() {
    global $pdo;
    
    $requestId = $_GET['request_id'] ?? null;
    $status = $_GET['status'] ?? 'approved';
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 20);
    $offset = ($page - 1) * $limit;
    
    if (!$requestId) {
        // Admin için tüm yorumları getir
        $whereClause = "WHERE 1=1";
        $params = [];
        
        if ($status !== 'all') {
            $whereClause .= " AND dc.status = ?";
            $params[] = $status;
        }
        
        $sql = "SELECT dc.*, du.username, du.full_name, dr.title as request_title,
                       au.full_name as approved_by_name
                FROM dua_comments dc
                JOIN dua_users du ON dc.user_id = du.id
                JOIN dua_requests dr ON dc.request_id = dr.id
                LEFT JOIN admin_users au ON dc.approved_by = au.id
                $whereClause
                ORDER BY dc.created_at DESC
                LIMIT ? OFFSET ?";
        
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $comments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Toplam sayı
        $countSql = "SELECT COUNT(*) FROM dua_comments dc $whereClause";
        $countStmt = $pdo->prepare($countSql);
        $countStmt->execute(array_slice($params, 0, -2));
        $total = $countStmt->fetchColumn();
        
        echo json_encode([
            'success' => true,
            'comments' => $comments,
            'total' => (int)$total,
            'page' => $page,
            'limit' => $limit
        ]);
    } else {
        // Belirli bir dua isteği için onaylanmış yorumları getir
        $sql = "SELECT dc.*, du.username, du.full_name
                FROM dua_comments dc
                JOIN dua_users du ON dc.user_id = du.id
                WHERE dc.request_id = ? AND dc.status = 'approved'
                ORDER BY dc.created_at ASC
                LIMIT ? OFFSET ?";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$requestId, $limit, $offset]);
        $comments = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Toplam sayı
        $countSql = "SELECT COUNT(*) FROM dua_comments WHERE request_id = ? AND status = 'approved'";
        $countStmt = $pdo->prepare($countSql);
        $countStmt->execute([$requestId]);
        $total = $countStmt->fetchColumn();
        
        echo json_encode([
            'success' => true,
            'comments' => $comments,
            'total' => (int)$total,
            'page' => $page,
            'limit' => $limit
        ]);
    }
}

function handleAddComment() {
    global $pdo, $input;
    
    $requestId = $input['request_id'] ?? null;
    $userId = $input['user_id'] ?? null;
    $comment = trim($input['comment'] ?? '');
    $commenterName = trim($input['commenter_name'] ?? '');
    $isAnonymous = (bool)($input['is_anonymous'] ?? false);
    
    // Validasyon
    if (!$requestId || !$userId || !$comment) {
        http_response_code(400);
        echo json_encode(['error' => 'Gerekli alanlar eksik']);
        return;
    }
    
    if (strlen($comment) < 10) {
        http_response_code(400);
        echo json_encode(['error' => 'Yorum en az 10 karakter olmalıdır']);
        return;
    }
    
    if (strlen($comment) > 1000) {
        http_response_code(400);
        echo json_encode(['error' => 'Yorum en fazla 1000 karakter olabilir']);
        return;
    }
    
    // Dua isteğinin var olduğunu kontrol et
    $checkSql = "SELECT id FROM dua_requests WHERE id = ? AND status = 'approved'";
    $checkStmt = $pdo->prepare($checkSql);
    $checkStmt->execute([$requestId]);
    if (!$checkStmt->fetch()) {
        http_response_code(404);
        echo json_encode(['error' => 'Dua isteği bulunamadı veya onaylanmamış']);
        return;
    }
    
    // Kullanıcının var olduğunu kontrol et
    $userCheckSql = "SELECT id FROM dua_users WHERE id = ? AND is_active = 1";
    $userCheckStmt = $pdo->prepare($userCheckSql);
    $userCheckStmt->execute([$userId]);
    if (!$userCheckStmt->fetch()) {
        http_response_code(404);
        echo json_encode(['error' => 'Kullanıcı bulunamadı veya aktif değil']);
        return;
    }
    
    // Spam kontrolü - aynı kullanıcı aynı dua isteğine 5 dakika içinde birden fazla yorum yapamaz
    $spamCheckSql = "SELECT id FROM dua_comments 
                     WHERE user_id = ? AND request_id = ? 
                     AND created_at > DATE_SUB(NOW(), INTERVAL 5 MINUTE)";
    $spamCheckStmt = $pdo->prepare($spamCheckSql);
    $spamCheckStmt->execute([$userId, $requestId]);
    if ($spamCheckStmt->fetch()) {
        http_response_code(429);
        echo json_encode(['error' => 'Çok sık yorum yapıyorsunuz. Lütfen 5 dakika bekleyin']);
        return;
    }
    
    // Anonim ise isim boş olmalı
    if ($isAnonymous) {
        $commenterName = '';
    }
    
    // Yorumu ekle
    $sql = "INSERT INTO dua_comments (request_id, user_id, comment, commenter_name, is_anonymous, status) 
            VALUES (?, ?, ?, ?, ?, 'pending')";
    
    $stmt = $pdo->prepare($sql);
    $result = $stmt->execute([$requestId, $userId, $comment, $commenterName, $isAnonymous ? 1 : 0]);
    
    if ($result) {
        $commentId = $pdo->lastInsertId();
        
        // Eklenen yorumu getir
        $getSql = "SELECT dc.*, du.username, du.full_name
                   FROM dua_comments dc
                   JOIN dua_users du ON dc.user_id = du.id
                   WHERE dc.id = ?";
        $getStmt = $pdo->prepare($getSql);
        $getStmt->execute([$commentId]);
        $newComment = $getStmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Yorumunuz başarıyla gönderildi. Admin onayından sonra yayınlanacaktır.',
            'comment' => $newComment
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Yorum eklenirken hata oluştu']);
    }
}

function handleUpdateComment() {
    global $pdo, $input;
    
    $commentId = $input['id'] ?? null;
    $status = $input['status'] ?? null;
    $adminNotes = trim($input['admin_notes'] ?? '');
    $adminId = $input['admin_id'] ?? null;
    
    if (!$commentId || !$status || !in_array($status, ['approved', 'rejected'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Geçersiz parametreler']);
        return;
    }
    
    // Admin kontrolü (gerçek uygulamada session kontrolü yapılmalı)
    if (!$adminId) {
        http_response_code(401);
        echo json_encode(['error' => 'Admin yetkisi gerekli']);
        return;
    }
    
    $sql = "UPDATE dua_comments 
            SET status = ?, admin_notes = ?, approved_by = ?, approved_at = NOW()
            WHERE id = ?";
    
    $stmt = $pdo->prepare($sql);
    $result = $stmt->execute([$status, $adminNotes, $adminId, $commentId]);
    
    if ($result) {
        echo json_encode([
            'success' => true,
            'message' => 'Yorum durumu güncellendi'
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Güncelleme sırasında hata oluştu']);
    }
}

function handleDeleteComment() {
    global $pdo;
    
    $commentId = $_GET['id'] ?? null;
    $adminId = $_GET['admin_id'] ?? null;
    
    if (!$commentId) {
        http_response_code(400);
        echo json_encode(['error' => 'Yorum ID gerekli']);
        return;
    }
    
    // Admin kontrolü (gerçek uygulamada session kontrolü yapılmalı)
    if (!$adminId) {
        http_response_code(401);
        echo json_encode(['error' => 'Admin yetkisi gerekli']);
        return;
    }
    
    $sql = "DELETE FROM dua_comments WHERE id = ?";
    $stmt = $pdo->prepare($sql);
    $result = $stmt->execute([$commentId]);
    
    if ($result) {
        echo json_encode([
            'success' => true,
            'message' => 'Yorum silindi'
        ]);
    } else {
        http_response_code(500);
        echo json_encode(['error' => 'Silme sırasında hata oluştu']);
    }
}
?>
