<?php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// OPTIONS isteği için
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    exit(0);
}

require_once '../config/database.php';

// Veritabanı bağlantısı (stories.php ile aynı mantık)
try {
    $pdo = getDB();
    
    if (!$pdo) {
        throw new Exception('Database connection failed');
    }
} catch(Exception $e) {
    error_log("Database connection error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Veritabanı bağlantı hatası: ' . $e->getMessage()
    ]);
    exit;
}

$action = $_GET['action'] ?? '';

switch($action) {
    case 'create':
        handleCreateRequest($pdo);
        break;
    case 'list':
        handleListRequests($pdo);
        break;
    case 'get':
        handleGetRequest($pdo);
        break;
    case 'update':
        handleUpdateRequest($pdo);
        break;
    case 'delete':
        handleDeleteRequest($pdo);
        break;
    case 'categories':
        handleGetCategories($pdo);
        break;
    case 'pray':
        handlePrayForRequest($pdo);
        break;
    case 'user_requests':
        handleUserRequests($pdo);
        break;
    case 'stats':
        handleGetStats($pdo);
        break;
    default:
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Geçersiz işlem'
        ]);
        break;
}

/**
 * Yeni dua talebi oluştur
 */
function handleCreateRequest($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece POST metodu desteklenir']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Gerekli alanları kontrol et
    $required_fields = ['user_id', 'category_id', 'title', 'description'];
    foreach ($required_fields as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => ucfirst($field) . ' alanı zorunludur'
            ]);
            return;
        }
    }
    
    $user_id = (int)$input['user_id'];
    $category_id = (int)$input['category_id'];
    $title = trim($input['title']);
    $description = trim($input['description']);
    $requester_name = trim($input['requester_name'] ?? '');
    $is_anonymous = (bool)($input['is_anonymous'] ?? false);
    
    // Validasyonlar
    if (strlen($title) < 5 || strlen($title) > 200) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Başlık 5-200 karakter arasında olmalıdır'
        ]);
        return;
    }
    
    if (strlen($description) < 10 || strlen($description) > 1000) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Açıklama 10-1000 karakter arasında olmalıdır'
        ]);
        return;
    }
    
    try {
        // Kullanıcı kontrolü
        $stmt = $pdo->prepare("SELECT id FROM dua_users WHERE id = ? AND is_active = 1");
        $stmt->execute([$user_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'message' => 'Geçersiz kullanıcı'
            ]);
            return;
        }
        
        // Kategori kontrolü
        $stmt = $pdo->prepare("SELECT id FROM dua_categories WHERE id = ?");
        $stmt->execute([$category_id]);
        if (!$stmt->fetch()) {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'message' => 'Geçersiz kategori'
            ]);
            return;
        }
        
        // Dua talebini kaydet
        $stmt = $pdo->prepare("
            INSERT INTO dua_requests (user_id, category_id, title, description, requester_name, is_anonymous) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $user_id,
            $category_id,
            $title,
            $description,
            $requester_name,
            $is_anonymous ? 1 : 0
        ]);
        
        $request_id = $pdo->lastInsertId();
        
        // Oluşturulan talebi getir
        $stmt = $pdo->prepare("
            SELECT dr.*, dc.name as category_name, du.username, du.full_name
            FROM dua_requests dr
            JOIN dua_categories dc ON dr.category_id = dc.id
            JOIN dua_users du ON dr.user_id = du.id
            WHERE dr.id = ?
        ");
        $stmt->execute([$request_id]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'message' => 'Dua talebi başarıyla oluşturuldu. Admin onayından sonra yayınlanacaktır.',
            'data' => $request
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Dua talebi oluşturulurken bir hata oluştu'
        ]);
    }
}

/**
 * Onaylanmış dua taleplerini listele
 */
function handleListRequests($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece GET metodu desteklenir']);
        return;
    }
    
    $category_id = $_GET['category_id'] ?? '';
    $page = max(1, (int)($_GET['page'] ?? 1));
    $limit = min(50, max(1, (int)($_GET['limit'] ?? 20)));
    $offset = ($page - 1) * $limit;
    
    try {
        // Debug logging
        error_log("handleListRequests called - category_id: $category_id, page: $page, limit: $limit");
        
        // WHERE koşulunu oluştur
        $where_conditions = ["dr.status = 'approved'"];
        $params = [];
        
        if (!empty($category_id)) {
            $where_conditions[] = "dr.category_id = ?";
            $params[] = $category_id;
        }
        
        $where_clause = implode(' AND ', $where_conditions);
        error_log("WHERE clause: $where_clause");
        error_log("Params: " . json_encode($params));
        
        // Toplam sayıyı al
        $count_sql = "
            SELECT COUNT(*) as total
            FROM dua_requests dr
            WHERE $where_clause
        ";
        $stmt = $pdo->prepare($count_sql);
        $stmt->execute($params);
        $total = $stmt->fetch(PDO::FETCH_ASSOC)['total'];
        error_log("Total count: $total");
        
        // Dua taleplerini getir
        $sql = "
            SELECT 
                dr.id,
                dr.title,
                dr.description,
                dr.requester_name,
                dr.is_anonymous,
                dr.prayer_count,
                dr.view_count,
                dr.created_at,
                dc.name as category_name,
                '#2196F3' as category_color,
                CASE 
                    WHEN dr.is_anonymous = 1 THEN 'Anonim'
                    WHEN dr.requester_name IS NOT NULL AND dr.requester_name != '' THEN dr.requester_name
                    ELSE du.full_name
                END as display_name
            FROM dua_requests dr
            JOIN dua_categories dc ON dr.category_id = dc.id
            JOIN dua_users du ON dr.user_id = du.id
            WHERE $where_clause
            ORDER BY dr.created_at DESC
            LIMIT ? OFFSET ?
        ";
        
        $params[] = $limit;
        $params[] = $offset;
        
        error_log("Final SQL: $sql");
        error_log("Final params: " . json_encode($params));
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $requests = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        error_log("Found " . count($requests) . " requests");
        
        // Sayfa bilgilerini hesapla
        $total_pages = ceil($total / $limit);
        
        $response = [
            'success' => true,
            'data' => [
                'requests' => $requests,
                'pagination' => [
                    'current_page' => $page,
                    'total_pages' => $total_pages,
                    'total_items' => $total,
                    'items_per_page' => $limit,
                    'has_next' => $page < $total_pages,
                    'has_prev' => $page > 1
                ]
            ]
        ];
        
        error_log("Response: " . json_encode($response));
        echo json_encode($response);
        
    } catch (PDOException $e) {
        error_log("PDO Error in handleListRequests: " . $e->getMessage());
        error_log("PDO Error Code: " . $e->getCode());
        error_log("PDO Error Info: " . print_r($e->errorInfo, true));
        
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Dua talepleri getirilirken bir hata oluştu: ' . $e->getMessage()
        ]);
    } catch (Exception $e) {
        error_log("General Error in handleListRequests: " . $e->getMessage());
        
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Beklenmeyen bir hata oluştu: ' . $e->getMessage()
        ]);
    }
}

/**
 * Belirli bir dua talebini getir
 */
function handleGetRequest($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece GET metodu desteklenir']);
        return;
    }
    
    $request_id = $_GET['id'] ?? '';
    
    if (empty($request_id)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Dua talebi ID gereklidir'
        ]);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                dr.*,
                dc.name as category_name,
                '#2196F3' as category_color,
                CASE 
                    WHEN dr.is_anonymous = 1 THEN 'Anonim'
                    WHEN dr.requester_name IS NOT NULL AND dr.requester_name != '' THEN dr.requester_name
                    ELSE du.full_name
                END as display_name
            FROM dua_requests dr
            JOIN dua_categories dc ON dr.category_id = dc.id
            JOIN dua_users du ON dr.user_id = du.id
            WHERE dr.id = ? AND dr.status = 'approved'
        ");
        $stmt->execute([$request_id]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$request) {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'message' => 'Dua talebi bulunamadı'
            ]);
            return;
        }
        
        // Görüntülenme sayısını artır
        $stmt = $pdo->prepare("UPDATE dua_requests SET view_count = view_count + 1 WHERE id = ?");
        $stmt->execute([$request_id]);
        $request['view_count']++;
        
        echo json_encode([
            'success' => true,
            'data' => $request
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Dua talebi getirilirken bir hata oluştu'
        ]);
    }
}

/**
 * Dua talebini güncelle (sadece kendi talepleri)
 */
function handleUpdateRequest($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'PUT') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece PUT metodu desteklenir']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['id']) || empty($input['user_id'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Dua talebi ID ve kullanıcı ID gereklidir'
        ]);
        return;
    }
    
    $request_id = (int)$input['id'];
    $user_id = (int)$input['user_id'];
    
    try {
        // Kullanıcının bu talebi güncelleme yetkisi var mı kontrol et
        $stmt = $pdo->prepare("
            SELECT id, status FROM dua_requests 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$request_id, $user_id]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$request) {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'message' => 'Dua talebi bulunamadı veya güncelleme yetkiniz yok'
            ]);
            return;
        }
        
        // Onaylanmış talepleri güncellemeyi engelle
        if ($request['status'] === 'approved') {
            http_response_code(403);
            echo json_encode([
                'success' => false,
                'message' => 'Onaylanmış dua talepleri güncellenemez'
            ]);
            return;
        }
        
        // Güncellenecek alanları belirle
        $update_fields = [];
        $params = [];
        
        if (!empty($input['title'])) {
            $title = trim($input['title']);
            if (strlen($title) >= 5 && strlen($title) <= 200) {
                $update_fields[] = "title = ?";
                $params[] = $title;
            }
        }
        
        if (!empty($input['description'])) {
            $description = trim($input['description']);
            if (strlen($description) >= 10 && strlen($description) <= 1000) {
                $update_fields[] = "description = ?";
                $params[] = $description;
            }
        }
        
        if (isset($input['category_id'])) {
            $update_fields[] = "category_id = ?";
            $params[] = (int)$input['category_id'];
        }
        
        if (isset($input['requester_name'])) {
            $update_fields[] = "requester_name = ?";
            $params[] = trim($input['requester_name']);
        }
        
        if (isset($input['is_anonymous'])) {
            $update_fields[] = "is_anonymous = ?";
            $params[] = (bool)$input['is_anonymous'] ? 1 : 0;
        }
        
        if (empty($update_fields)) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'Güncellenecek geçerli alan bulunamadı'
            ]);
            return;
        }
        
        // Güncelleme yap
        $update_fields[] = "updated_at = NOW()";
        $params[] = $request_id;
        
        $sql = "UPDATE dua_requests SET " . implode(', ', $update_fields) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        echo json_encode([
            'success' => true,
            'message' => 'Dua talebi başarıyla güncellendi'
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Dua talebi güncellenirken bir hata oluştu'
        ]);
    }
}

/**
 * Dua talebini sil (sadece kendi talepleri)
 */
function handleDeleteRequest($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'DELETE') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece DELETE metodu desteklenir']);
        return;
    }
    
    $request_id = $_GET['id'] ?? '';
    $user_id = $_GET['user_id'] ?? '';
    
    if (empty($request_id) || empty($user_id)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Dua talebi ID ve kullanıcı ID gereklidir'
        ]);
        return;
    }
    
    try {
        // Kullanıcının bu talebi silme yetkisi var mı kontrol et
        $stmt = $pdo->prepare("
            SELECT id, status FROM dua_requests 
            WHERE id = ? AND user_id = ?
        ");
        $stmt->execute([$request_id, $user_id]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$request) {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'message' => 'Dua talebi bulunamadı veya silme yetkiniz yok'
            ]);
            return;
        }
        
        // Onaylanmış talepleri silmeyi engelle
        if ($request['status'] === 'approved') {
            http_response_code(403);
            echo json_encode([
                'success' => false,
                'message' => 'Onaylanmış dua talepleri silinemez'
            ]);
            return;
        }
        
        // Talebi sil
        $stmt = $pdo->prepare("DELETE FROM dua_requests WHERE id = ?");
        $stmt->execute([$request_id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Dua talebi başarıyla silindi'
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Dua talebi silinirken bir hata oluştu'
        ]);
    }
}

/**
 * Kategorileri getir
 */
function handleGetCategories($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece GET metodu desteklenir']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT dc.*, COUNT(dr.id) as request_count
            FROM dua_categories dc
            LEFT JOIN dua_requests dr ON dc.id = dr.category_id AND dr.status = 'approved'
            GROUP BY dc.id
            ORDER BY dc.sort_order ASC, dc.name ASC
        ");
        $stmt->execute();
        $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $categories
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Kategoriler getirilirken bir hata oluştu'
        ]);
    }
}

/**
 * Dua talebi için dua et (sayacı artır)
 */
function handlePrayForRequest($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece POST metodu desteklenir']);
        return;
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (empty($input['request_id'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Dua talebi ID gereklidir'
        ]);
        return;
    }
    
    $request_id = (int)$input['request_id'];
    
    try {
        // Dua talebinin varlığını kontrol et
        $stmt = $pdo->prepare("
            SELECT id, prayer_count FROM dua_requests 
            WHERE id = ? AND status = 'approved'
        ");
        $stmt->execute([$request_id]);
        $request = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$request) {
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'message' => 'Dua talebi bulunamadı'
            ]);
            return;
        }
        
        // Dua sayısını artır
        $stmt = $pdo->prepare("UPDATE dua_requests SET prayer_count = prayer_count + 1 WHERE id = ?");
        $stmt->execute([$request_id]);
        
        $new_count = $request['prayer_count'] + 1;
        
        echo json_encode([
            'success' => true,
            'message' => 'Dua edildi',
            'data' => [
                'prayer_count' => $new_count
            ]
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Dua sayısı güncellenirken bir hata oluştu'
        ]);
    }
}

/**
 * Kullanıcının kendi dua taleplerini getir
 */
function handleUserRequests($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece GET metodu desteklenir']);
        return;
    }
    
    $user_id = $_GET['user_id'] ?? '';
    
    if (empty($user_id)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Kullanıcı ID gereklidir'
        ]);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                dr.*,
                dc.name as category_name,
                '#2196F3' as category_color
            FROM dua_requests dr
            JOIN dua_categories dc ON dr.category_id = dc.id
            WHERE dr.user_id = ?
            ORDER BY dr.created_at DESC
        ");
        $stmt->execute([$user_id]);
        $requests = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => $requests
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Kullanıcı talepleri getirilirken bir hata oluştu'
        ]);
    }
}

/**
 * İstatistikleri getir
 */
function handleGetStats($pdo) {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        http_response_code(405);
        echo json_encode(['success' => false, 'message' => 'Sadece GET metodu desteklenir']);
        return;
    }
    
    try {
        // Genel istatistikler
        $stmt = $pdo->prepare("
            SELECT 
                COUNT(*) as total_requests,
                SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_requests,
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_requests,
                SUM(prayer_count) as total_prayers,
                SUM(view_count) as total_views
            FROM dua_requests
        ");
        $stmt->execute();
        $general_stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Kategori bazında istatistikler
        $stmt = $pdo->prepare("
            SELECT 
                dc.name as category_name,
                '#2196F3' as category_color,
                COUNT(dr.id) as request_count,
                SUM(dr.prayer_count) as prayer_count
            FROM dua_categories dc
            LEFT JOIN dua_requests dr ON dc.id = dr.category_id AND dr.status = 'approved'
            GROUP BY dc.id
            ORDER BY request_count DESC
        ");
        $stmt->execute();
        $category_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'data' => [
                'general' => $general_stats,
                'categories' => $category_stats
            ]
        ]);
        
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'İstatistikler getirilirken bir hata oluştu'
        ]);
    }
}
?>
