<?php
/**
 * Şifre Değiştirme Sayfası
 * Admin Panel - Admin şifresini değiştirme
 */

session_start();
require_once __DIR__ . '/config/database.php';

$page_title = 'Şifre Değiştirme';

// Admin session kontrolü
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

// Database bağlantısı
$database = new Database();
$db = $database->getConnection();

// Form işleme
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $current_password = $_POST['current_password'] ?? '';
        $new_password = $_POST['new_password'] ?? '';
        $confirm_password = $_POST['confirm_password'] ?? '';
        
        // Validasyon
        if (empty($current_password)) {
            throw new Exception("Mevcut şifre gereklidir");
        }
        
        if (empty($new_password)) {
            throw new Exception("Yeni şifre gereklidir");
        }
        
        if (strlen($new_password) < 6) {
            throw new Exception("Yeni şifre en az 6 karakter olmalıdır");
        }
        
        if ($new_password !== $confirm_password) {
            throw new Exception("Yeni şifre ve şifre tekrarı eşleşmiyor");
        }
        
        // Mevcut şifreyi kontrol et
        $stmt = $db->prepare("SELECT password FROM admin_users WHERE id = ?");
        $stmt->execute([$_SESSION['admin_id']]);
        $admin = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$admin || md5($current_password) !== $admin['password']) {
            throw new Exception("Mevcut şifre hatalı");
        }
        
        // Yeni şifreyi güncelle
        $new_password_hash = md5($new_password);
        $stmt = $db->prepare("UPDATE admin_users SET password = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?");
        
        if ($stmt->execute([$new_password_hash, $_SESSION['admin_id']])) {
            $_SESSION['success_message'] = 'Şifreniz başarıyla değiştirildi';
        } else {
            throw new Exception("Şifre güncellenirken hata oluştu");
        }
        
        header('Location: change-password.php');
        exit();
        
    } catch (Exception $e) {
        $_SESSION['error_message'] = $e->getMessage();
    }
}

include 'includes/header.php';
?>

<div class="page-header">
    <h2>Şifre Değiştirme</h2>
    <p>Admin hesabınızın şifresini güvenli bir şekilde değiştirin</p>
</div>

<!-- Şifre Değiştirme Formu -->
<div class="password-change-container">
    <div class="password-card">
        <div class="card-header">
            <h3>
                <i class="fas fa-shield-alt"></i>
                Şifre Güvenliği
            </h3>
        </div>
        
        <div class="card-body">
            <!-- Güvenlik Bilgilendirmesi -->
            <div class="security-info">
                <div class="info-item">
                    <i class="fas fa-info-circle"></i>
                    <div class="info-content">
                        <h4>Güvenli Şifre Önerileri</h4>
                        <ul>
                            <li>En az 8 karakter uzunluğunda olmalı</li>
                            <li>Büyük ve küçük harfler içermeli</li>
                            <li>Rakam ve özel karakterler kullanmalı</li>
                            <li>Kişisel bilgilerinizi içermemeli</li>
                            <li>Düzenli olarak değiştirilmeli</li>
                        </ul>
                    </div>
                </div>
            </div>
            
            <form method="POST" class="password-form" id="passwordForm">
                <div class="form-group">
                    <label for="current_password">
                        <i class="fas fa-key"></i>
                        Mevcut Şifre *
                    </label>
                    <div class="password-input-group">
                        <input type="password" id="current_password" name="current_password" 
                               class="form-control" required 
                               placeholder="Mevcut şifrenizi girin">
                        <button type="button" class="password-toggle" onclick="togglePassword('current_password')">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="new_password">
                        <i class="fas fa-lock"></i>
                        Yeni Şifre *
                    </label>
                    <div class="password-input-group">
                        <input type="password" id="new_password" name="new_password" 
                               class="form-control" required minlength="6"
                               placeholder="Yeni şifrenizi girin"
                               onkeyup="checkPasswordStrength()">
                        <button type="button" class="password-toggle" onclick="togglePassword('new_password')">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    <div class="password-strength" id="passwordStrength">
                        <div class="strength-bar">
                            <div class="strength-fill" id="strengthFill"></div>
                        </div>
                        <div class="strength-text" id="strengthText">Şifre gücü</div>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="confirm_password">
                        <i class="fas fa-check-double"></i>
                        Yeni Şifre Tekrar *
                    </label>
                    <div class="password-input-group">
                        <input type="password" id="confirm_password" name="confirm_password" 
                               class="form-control" required 
                               placeholder="Yeni şifrenizi tekrar girin"
                               onkeyup="checkPasswordMatch()">
                        <button type="button" class="password-toggle" onclick="togglePassword('confirm_password')">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                    <div class="password-match" id="passwordMatch"></div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary" id="submitBtn">
                        <i class="fas fa-save"></i>
                        Şifreyi Değiştir
                    </button>
                    <button type="reset" class="btn btn-secondary" onclick="resetForm()">
                        <i class="fas fa-undo"></i>
                        Temizle
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Güvenlik İpuçları -->
    <div class="security-tips">
        <div class="card-header">
            <h3>
                <i class="fas fa-lightbulb"></i>
                Güvenlik İpuçları
            </h3>
        </div>
        
        <div class="card-body">
            <div class="tip-item">
                <i class="fas fa-clock"></i>
                <div class="tip-content">
                    <h4>Düzenli Değiştirin</h4>
                    <p>Şifrenizi 3-6 ayda bir değiştirmeniz önerilir.</p>
                </div>
            </div>
            
            <div class="tip-item">
                <i class="fas fa-user-secret"></i>
                <div class="tip-content">
                    <h4>Gizli Tutun</h4>
                    <p>Şifrenizi kimseyle paylaşmayın ve güvenli bir yerde saklayın.</p>
                </div>
            </div>
            
            <div class="tip-item">
                <i class="fas fa-wifi"></i>
                <div class="tip-content">
                    <h4>Güvenli Bağlantı</h4>
                    <p>Şifre değiştirirken güvenli bir internet bağlantısı kullanın.</p>
                </div>
            </div>
            
            <div class="tip-item">
                <i class="fas fa-sign-out-alt"></i>
                <div class="tip-content">
                    <h4>Oturumu Kapatın</h4>
                    <p>İşiniz bittiğinde mutlaka oturumu kapatın.</p>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Şifre görünürlüğünü değiştir
function togglePassword(inputId) {
    const input = document.getElementById(inputId);
    const button = input.nextElementSibling;
    const icon = button.querySelector('i');
    
    if (input.type === 'password') {
        input.type = 'text';
        icon.className = 'fas fa-eye-slash';
    } else {
        input.type = 'password';
        icon.className = 'fas fa-eye';
    }
}

// Şifre gücünü kontrol et
function checkPasswordStrength() {
    const password = document.getElementById('new_password').value;
    const strengthFill = document.getElementById('strengthFill');
    const strengthText = document.getElementById('strengthText');
    
    let strength = 0;
    let strengthLabel = '';
    let strengthColor = '';
    
    if (password.length >= 6) strength += 1;
    if (password.length >= 8) strength += 1;
    if (/[a-z]/.test(password)) strength += 1;
    if (/[A-Z]/.test(password)) strength += 1;
    if (/[0-9]/.test(password)) strength += 1;
    if (/[^A-Za-z0-9]/.test(password)) strength += 1;
    
    switch (strength) {
        case 0:
        case 1:
            strengthLabel = 'Çok Zayıf';
            strengthColor = '#e74c3c';
            break;
        case 2:
            strengthLabel = 'Zayıf';
            strengthColor = '#f39c12';
            break;
        case 3:
        case 4:
            strengthLabel = 'Orta';
            strengthColor = '#f1c40f';
            break;
        case 5:
            strengthLabel = 'Güçlü';
            strengthColor = '#2ecc71';
            break;
        case 6:
            strengthLabel = 'Çok Güçlü';
            strengthColor = '#27ae60';
            break;
    }
    
    const percentage = (strength / 6) * 100;
    strengthFill.style.width = percentage + '%';
    strengthFill.style.backgroundColor = strengthColor;
    strengthText.textContent = strengthLabel;
    strengthText.style.color = strengthColor;
}

// Şifre eşleşmesini kontrol et
function checkPasswordMatch() {
    const newPassword = document.getElementById('new_password').value;
    const confirmPassword = document.getElementById('confirm_password').value;
    const matchDiv = document.getElementById('passwordMatch');
    
    if (confirmPassword === '') {
        matchDiv.textContent = '';
        return;
    }
    
    if (newPassword === confirmPassword) {
        matchDiv.textContent = '✓ Şifreler eşleşiyor';
        matchDiv.style.color = '#27ae60';
    } else {
        matchDiv.textContent = '✗ Şifreler eşleşmiyor';
        matchDiv.style.color = '#e74c3c';
    }
}

// Formu temizle
function resetForm() {
    document.getElementById('passwordForm').reset();
    document.getElementById('passwordStrength').style.display = 'none';
    document.getElementById('passwordMatch').textContent = '';
}

// Form validasyonu
document.getElementById('passwordForm').addEventListener('submit', function(e) {
    const newPassword = document.getElementById('new_password').value;
    const confirmPassword = document.getElementById('confirm_password').value;
    
    if (newPassword !== confirmPassword) {
        e.preventDefault();
        alert('Yeni şifre ve şifre tekrarı eşleşmiyor!');
        return false;
    }
    
    if (newPassword.length < 6) {
        e.preventDefault();
        alert('Yeni şifre en az 6 karakter olmalıdır!');
        return false;
    }
    
    return confirm('Şifrenizi değiştirmek istediğinizden emin misiniz?');
});

// Yeni şifre alanına odaklanıldığında şifre gücü göstergesi göster
document.getElementById('new_password').addEventListener('focus', function() {
    document.getElementById('passwordStrength').style.display = 'block';
});
</script>

<style>
.password-change-container {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 2rem;
    margin-top: 1rem;
}

.password-card, .security-tips {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    overflow: hidden;
}

.card-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.card-header h3 {
    margin: 0;
    font-size: 1.25rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.card-body {
    padding: 2rem;
}

.security-info {
    background: #f8f9fa;
    border-radius: 8px;
    padding: 1.5rem;
    margin-bottom: 2rem;
    border-left: 4px solid #4f46e5;
}

.info-item {
    display: flex;
    gap: 1rem;
    align-items: flex-start;
}

.info-item i {
    color: #4f46e5;
    font-size: 1.5rem;
    margin-top: 0.25rem;
}

.info-content h4 {
    margin: 0 0 0.5rem 0;
    color: #1f2937;
    font-size: 1.1rem;
}

.info-content ul {
    margin: 0;
    padding-left: 1.5rem;
    color: #6b7280;
}

.info-content li {
    margin-bottom: 0.25rem;
    line-height: 1.5;
}

.password-form {
    max-width: 500px;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #374151;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.password-input-group {
    position: relative;
    display: flex;
}

.password-input-group input {
    flex: 1;
    padding: 0.75rem;
    border: 2px solid #e5e7eb;
    border-radius: 8px 0 0 8px;
    font-size: 0.9rem;
    transition: border-color 0.3s;
}

.password-input-group input:focus {
    outline: none;
    border-color: #4f46e5;
}

.password-toggle {
    padding: 0.75rem 1rem;
    border: 2px solid #e5e7eb;
    border-left: none;
    border-radius: 0 8px 8px 0;
    background: #f9fafb;
    color: #6b7280;
    cursor: pointer;
    transition: all 0.3s;
}

.password-toggle:hover {
    background: #f3f4f6;
    color: #374151;
}

.password-strength {
    margin-top: 0.5rem;
    display: none;
}

.strength-bar {
    width: 100%;
    height: 6px;
    background: #e5e7eb;
    border-radius: 3px;
    overflow: hidden;
    margin-bottom: 0.5rem;
}

.strength-fill {
    height: 100%;
    width: 0%;
    transition: all 0.3s ease;
    border-radius: 3px;
}

.strength-text {
    font-size: 0.8rem;
    font-weight: 600;
}

.password-match {
    margin-top: 0.5rem;
    font-size: 0.9rem;
    font-weight: 500;
}

.form-actions {
    display: flex;
    gap: 1rem;
    margin-top: 2rem;
}

.btn {
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s;
    text-decoration: none;
}

.btn-primary {
    background: linear-gradient(135deg, #4f46e5, #7c3aed);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(79, 70, 229, 0.4);
}

.btn-secondary {
    background: #f3f4f6;
    color: #374151;
}

.btn-secondary:hover {
    background: #e5e7eb;
}

.security-tips .card-body {
    padding: 1.5rem;
}

.tip-item {
    display: flex;
    gap: 1rem;
    align-items: flex-start;
    margin-bottom: 1.5rem;
    padding: 1rem;
    background: #f8f9fa;
    border-radius: 8px;
    border-left: 3px solid #4f46e5;
}

.tip-item:last-child {
    margin-bottom: 0;
}

.tip-item i {
    color: #4f46e5;
    font-size: 1.25rem;
    margin-top: 0.25rem;
    flex-shrink: 0;
}

.tip-content h4 {
    margin: 0 0 0.5rem 0;
    color: #1f2937;
    font-size: 1rem;
}

.tip-content p {
    margin: 0;
    color: #6b7280;
    font-size: 0.9rem;
    line-height: 1.5;
}

@media (max-width: 768px) {
    .password-change-container {
        grid-template-columns: 1fr;
    }
    
    .form-actions {
        flex-direction: column;
    }
    
    .password-form {
        max-width: none;
    }
}
</style>

<?php include 'includes/footer.php'; ?>
