<?php
/**
 * Story Düzenleme Sayfası
 * Namaz Vakitleri Admin Panel
 */

$page_title = 'Story Düzenle';
$header_actions = '<a href="story-cards.php" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Geri Dön</a>';

require_once 'config/database.php';
require_once 'includes/header.php';

// Story ID kontrolü
$story_id = (int)($_GET['id'] ?? 0);
if ($story_id <= 0) {
    $_SESSION['error_message'] = 'Geçersiz story ID.';
    header('Location: story-cards.php');
    exit();
}

// Story bilgilerini al
try {
    $db = getDB();
    $stmt = $db->prepare("SELECT * FROM story_cards WHERE id = ?");
    $stmt->execute([$story_id]);
    $story = $stmt->fetch();
    
    if (!$story) {
        $_SESSION['error_message'] = 'Story bulunamadı.';
        header('Location: story-cards.php');
        exit();
    }
} catch (PDOException $e) {
    $_SESSION['error_message'] = 'Story yüklenirken hata oluştu.';
    header('Location: story-cards.php');
    exit();
}

// Form gönderildi mi?
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = trim($_POST['title'] ?? '');
    $content = trim($_POST['content'] ?? '');
    $category_id = (int)($_POST['category_id'] ?? 0);
    $content_type = $_POST['content_type'] ?? 'text';
    $background_color = $_POST['background_color'] ?? '#1a1a1a';
    $text_color = $_POST['text_color'] ?? '#ffffff';
    $font_size = (int)($_POST['font_size'] ?? 16);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    $errors = [];
    
    // Validasyon
    if (empty($title)) {
        $errors[] = 'Başlık gereklidir.';
    }
    
    if (empty($content) && $content_type === 'text') {
        $errors[] = 'Metin içeriği gereklidir.';
    }
    
    if ($category_id <= 0) {
        $errors[] = 'Kategori seçimi gereklidir.';
    }
    
    // Görsel yükleme kontrolü
    $image_path = $story['image_path']; // Mevcut görsel
    $delete_old_image = false;
    
    if ($content_type === 'image' && isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $upload_dir = 'uploads/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        $file_info = pathinfo($_FILES['image']['name']);
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        $file_extension = strtolower($file_info['extension']);
        
        if (!in_array($file_extension, $allowed_extensions)) {
            $errors[] = 'Sadece JPG, PNG, GIF ve WebP formatları desteklenir.';
        } else {
            // Dosya boyutu kontrolü (5MB)
            if ($_FILES['image']['size'] > 5 * 1024 * 1024) {
                $errors[] = 'Dosya boyutu 5MB\'dan küçük olmalıdır.';
            } else {
                // Görsel boyutu kontrolü
                $image_info = getimagesize($_FILES['image']['tmp_name']);
                if ($image_info === false) {
                    $errors[] = 'Geçersiz görsel dosyası.';
                } else {
                    $width = $image_info[0];
                    $height = $image_info[1];
                    
                    // 1080x1920 boyut kontrolü (tolerans %10)
                    if (abs($width - 1080) > 108 || abs($height - 1920) > 192) {
                        $errors[] = 'Görsel boyutu 1080x1920 piksel olmalıdır. (Mevcut: ' . $width . 'x' . $height . ')';
                    } else {
                        // Dosya adını oluştur
                        $filename = uniqid() . '_' . time() . '.' . $file_extension;
                        $target_path = $upload_dir . $filename;
                        
                        if (move_uploaded_file($_FILES['image']['tmp_name'], $target_path)) {
                            $delete_old_image = true;
                            $image_path = $filename;
                        } else {
                            $errors[] = 'Dosya yüklenirken hata oluştu.';
                        }
                    }
                }
            }
        }
    } elseif ($content_type === 'image' && empty($story['image_path'])) {
        $errors[] = 'Görsel dosyası seçmelisiniz.';
    }
    
    // İçerik türü text'e değiştirilirse görsel silinsin
    if ($content_type === 'text' && !empty($story['image_path'])) {
        $delete_old_image = true;
        $image_path = null;
    }
    
    // Hata yoksa güncelle
    if (empty($errors)) {
        try {
            $db = getDB();
            
            $sql = "UPDATE story_cards SET 
                category_id = ?, title = ?, content = ?, content_type = ?, image_path = ?, 
                background_color = ?, text_color = ?, font_size = ?, is_active = ?, updated_at = NOW()
                WHERE id = ?";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                $category_id, $title, $content, $content_type, $image_path,
                $background_color, $text_color, $font_size, $is_active, $story_id
            ]);
            
            // Eski görseli sil
            if ($delete_old_image && !empty($story['image_path']) && file_exists('uploads/' . $story['image_path'])) {
                unlink('uploads/' . $story['image_path']);
            }
            
            $_SESSION['success_message'] = 'Story başarıyla güncellendi.';
            header('Location: story-cards.php');
            exit();
            
        } catch (PDOException $e) {
            $errors[] = 'Veritabanı hatası: ' . $e->getMessage();
        }
    }
}

// Kategorileri al
try {
    $db = getDB();
    $stmt = $db->query("SELECT * FROM story_categories WHERE is_active = 1 ORDER BY sort_order");
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $categories = [];
    $_SESSION['error_message'] = 'Kategoriler yüklenirken hata oluştu.';
}

// Form verilerini ayarla (POST varsa POST'tan, yoksa veritabanından)
$form_data = [
    'title' => $_POST['title'] ?? $story['title'],
    'content' => $_POST['content'] ?? $story['content'],
    'category_id' => $_POST['category_id'] ?? $story['category_id'],
    'content_type' => $_POST['content_type'] ?? $story['content_type'],
    'background_color' => $_POST['background_color'] ?? $story['background_color'],
    'text_color' => $_POST['text_color'] ?? $story['text_color'],
    'font_size' => $_POST['font_size'] ?? $story['font_size'],
    'is_active' => isset($_POST['is_active']) ? $_POST['is_active'] : $story['is_active']
];
?>

<!-- Hata mesajları -->
<?php if (!empty($errors)): ?>
    <div class="alert alert-error">
        <ul style="margin: 0; padding-left: 20px;">
            <?php foreach ($errors as $error): ?>
                <li><?php echo htmlspecialchars($error); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<form method="POST" enctype="multipart/form-data" id="storyForm">
    <div class="row">
        <!-- Sol Kolon - Form Alanları -->
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h3><i class="fas fa-edit"></i> Story Bilgileri</h3>
                </div>
                <div class="card-body">
                    <!-- Başlık -->
                    <div class="form-group">
                        <label for="title" class="form-label">
                            <i class="fas fa-heading"></i> Başlık *
                        </label>
                        <input type="text" id="title" name="title" class="form-control" 
                               value="<?php echo htmlspecialchars($form_data['title']); ?>" 
                               required maxlength="200" data-max-length="200">
                    </div>
                    
                    <!-- Kategori -->
                    <div class="form-group">
                        <label for="category_id" class="form-label">
                            <i class="fas fa-tags"></i> Kategori *
                        </label>
                        <select id="category_id" name="category_id" class="form-control" required>
                            <option value="">Kategori Seçin</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>"
                                        <?php echo ($form_data['category_id'] == $category['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($category['display_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- İçerik Türü -->
                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-layer-group"></i> İçerik Türü *
                        </label>
                        <div style="display: flex; gap: 20px; margin-top: 10px;">
                            <label style="display: flex; align-items: center; cursor: pointer;">
                                <input type="radio" name="content_type" value="text" 
                                       <?php echo ($form_data['content_type'] === 'text') ? 'checked' : ''; ?>
                                       style="margin-right: 8px;">
                                <i class="fas fa-font" style="margin-right: 5px;"></i> Metin
                            </label>
                            <label style="display: flex; align-items: center; cursor: pointer;">
                                <input type="radio" name="content_type" value="image" 
                                       <?php echo ($form_data['content_type'] === 'image') ? 'checked' : ''; ?>
                                       style="margin-right: 8px;">
                                <i class="fas fa-image" style="margin-right: 5px;"></i> Görsel
                            </label>
                        </div>
                    </div>
                    
                    <!-- Metin İçeriği -->
                    <div class="form-group" id="textContent">
                        <label for="content" class="form-label">
                            <i class="fas fa-align-left"></i> İçerik *
                        </label>
                        <textarea id="content" name="content" class="form-control" rows="6" 
                                  maxlength="1000" data-max-length="1000"><?php echo htmlspecialchars($form_data['content']); ?></textarea>
                        <small class="text-muted">Maksimum 1000 karakter</small>
                    </div>
                    
                    <!-- Görsel Yükleme -->
                    <div class="form-group" id="imageContent" style="display: none;">
                        <label for="image" class="form-label">
                            <i class="fas fa-upload"></i> Görsel Dosyası (1080x1920 piksel)
                        </label>
                        
                        <?php if (!empty($story['image_path']) && file_exists('uploads/' . $story['image_path'])): ?>
                            <div class="current-image" style="margin-bottom: 15px;">
                                <p><strong>Mevcut Görsel:</strong></p>
                                <img src="uploads/<?php echo htmlspecialchars($story['image_path']); ?>" 
                                     style="max-width: 200px; max-height: 300px; border-radius: 8px; border: 2px solid #ddd;">
                                <p><small class="text-muted">Yeni görsel yüklerseniz mevcut görsel değiştirilecektir.</small></p>
                            </div>
                        <?php endif; ?>
                        
                        <div class="file-upload">
                            <input type="file" id="image" name="image" accept="image/*" 
                                   onchange="previewImage(this, 'imagePreview')">
                            <label for="image" class="file-upload-label">
                                <i class="fas fa-cloud-upload-alt" style="font-size: 24px; margin-bottom: 10px;"></i><br>
                                Yeni görsel dosyasını seçin veya sürükleyip bırakın<br>
                                <small>JPG, PNG, GIF, WebP - Maksimum 5MB</small>
                            </label>
                        </div>
                        <img id="imagePreview" class="image-preview" style="display: none;">
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Sağ Kolon - Tasarım ve Ayarlar -->
        <div class="col-md-4">
            <!-- Tasarım Ayarları -->
            <div class="card">
                <div class="card-header">
                    <h3><i class="fas fa-palette"></i> Tasarım Ayarları</h3>
                </div>
                <div class="card-body">
                    <!-- Arka Plan Rengi -->
                    <div class="form-group">
                        <label for="background_color" class="form-label">
                            <i class="fas fa-fill-drip"></i> Arka Plan Rengi
                        </label>
                        <input type="color" id="background_color" name="background_color" 
                               class="form-control" style="height: 50px;"
                               value="<?php echo $form_data['background_color']; ?>">
                    </div>
                    
                    <!-- Metin Rengi -->
                    <div class="form-group">
                        <label for="text_color" class="form-label">
                            <i class="fas fa-font"></i> Metin Rengi
                        </label>
                        <input type="color" id="text_color" name="text_color" 
                               class="form-control" style="height: 50px;"
                               value="<?php echo $form_data['text_color']; ?>">
                    </div>
                    
                    <!-- Font Boyutu -->
                    <div class="form-group">
                        <label for="font_size" class="form-label">
                            <i class="fas fa-text-height"></i> Font Boyutu
                        </label>
                        <select id="font_size" name="font_size" class="form-control">
                            <?php for ($size = 12; $size <= 24; $size += 2): ?>
                                <option value="<?php echo $size; ?>"
                                        <?php echo ($form_data['font_size'] == $size) ? 'selected' : ''; ?>>
                                    <?php echo $size; ?>px
                                </option>
                            <?php endfor; ?>
                        </select>
                    </div>
                </div>
            </div>
            
            <!-- Durum ve İşlemler -->
            <div class="card">
                <div class="card-header">
                    <h3><i class="fas fa-cog"></i> Durum ve İşlemler</h3>
                </div>
                <div class="card-body">
                    <!-- Aktif/Pasif -->
                    <div class="form-group">
                        <label style="display: flex; align-items: center; cursor: pointer;">
                            <input type="checkbox" name="is_active" value="1" 
                                   <?php echo $form_data['is_active'] ? 'checked' : ''; ?>
                                   style="margin-right: 10px;">
                            <i class="fas fa-eye" style="margin-right: 5px;"></i>
                            Story'yi aktif olarak yayınla
                        </label>
                    </div>
                    
                    <!-- Story Bilgileri -->
                    <div class="info-box" style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin: 20px 0;">
                        <h4 style="margin: 0 0 10px 0; font-size: 14px; color: #666;">Story Bilgileri</h4>
                        <p style="margin: 5px 0; font-size: 12px;"><strong>ID:</strong> <?php echo $story['id']; ?></p>
                        <p style="margin: 5px 0; font-size: 12px;"><strong>Oluşturulma:</strong> <?php echo date('d.m.Y H:i', strtotime($story['created_at'])); ?></p>
                        <?php if ($story['updated_at']): ?>
                            <p style="margin: 5px 0; font-size: 12px;"><strong>Son Güncelleme:</strong> <?php echo date('d.m.Y H:i', strtotime($story['updated_at'])); ?></p>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Butonlar -->
                    <div style="display: grid; gap: 10px; margin-top: 30px;">
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-save"></i> Değişiklikleri Kaydet
                        </button>
                        
                        <a href="story-cards.php" class="btn btn-secondary">
                            <i class="fas fa-times"></i> İptal Et
                        </a>
                        
                        <button type="button" class="btn btn-danger" onclick="deleteStory(<?php echo $story['id']; ?>)">
                            <i class="fas fa-trash"></i> Story'yi Sil
                        </button>
                    </div>
                </div>
            </div>
            
            <!-- Önizleme -->
            <div class="card" id="previewCard">
                <div class="card-header">
                    <h3><i class="fas fa-eye"></i> Önizleme</h3>
                </div>
                <div class="card-body">
                    <div id="storyPreview" style="
                        width: 200px; 
                        height: 355px; 
                        border-radius: 10px; 
                        padding: 20px; 
                        display: flex; 
                        align-items: center; 
                        justify-content: center; 
                        text-align: center;
                        background: <?php echo $form_data['background_color']; ?>;
                        color: <?php echo $form_data['text_color']; ?>;
                        font-size: <?php echo $form_data['font_size']; ?>px;
                        line-height: 1.4;
                        overflow: hidden;
                        margin: 0 auto;
                    ">
                        <div>Story önizlemesi burada görünecek</div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</form>

<script>
$(document).ready(function() {
    // İçerik türü değiştiğinde
    $('input[name="content_type"]').change(function() {
        var contentType = $(this).val();
        
        if (contentType === 'text') {
            $('#textContent').show();
            $('#imageContent').hide();
            $('#content').prop('required', true);
            $('#image').prop('required', false);
        } else {
            $('#textContent').hide();
            $('#imageContent').show();
            $('#content').prop('required', false);
            $('#image').prop('required', false); // Mevcut görsel olabilir
        }
        
        updatePreview();
    });
    
    // Sayfa yüklendiğinde doğru alanları göster
    var initialContentType = $('input[name="content_type"]:checked').val();
    if (initialContentType === 'text') {
        $('#textContent').show();
        $('#imageContent').hide();
        $('#content').prop('required', true);
        $('#image').prop('required', false);
    } else {
        $('#textContent').hide();
        $('#imageContent').show();
        $('#content').prop('required', false);
        $('#image').prop('required', false);
    }
    
    // Önizleme güncelleme
    function updatePreview() {
        var title = $('#title').val();
        var content = $('#content').val();
        var contentType = $('input[name="content_type"]:checked').val();
        var backgroundColor = $('#background_color').val();
        var textColor = $('#text_color').val();
        var fontSize = $('#font_size').val();
        
        var preview = $('#storyPreview');
        preview.css({
            'background': backgroundColor,
            'color': textColor,
            'font-size': fontSize + 'px'
        });
        
        if (contentType === 'text') {
            var previewText = title ? '<strong>' + title + '</strong><br><br>' + content : 'Story önizlemesi burada görünecek';
            preview.html('<div>' + previewText + '</div>');
        } else {
            preview.html('<div><i class="fas fa-image" style="font-size: 48px; margin-bottom: 10px;"></i><br>Görsel yüklendiğinde<br>burada görünecek</div>');
        }
    }
    
    // Form alanları değiştiğinde önizlemeyi güncelle
    $('#title, #content, #background_color, #text_color, #font_size').on('input change', updatePreview);
    
    // Sayfa yüklendiğinde önizlemeyi güncelle
    updatePreview();
    
    // Form validasyonu
    $('#storyForm').on('submit', function(e) {
        if (!validateForm('storyForm')) {
            e.preventDefault();
            alert('Lütfen tüm gerekli alanları doldurun.');
        }
    });
});

// Story silme fonksiyonu
function deleteStory(storyId) {
    if (confirm('Bu story\'yi kalıcı olarak silmek istediğinizden emin misiniz?\n\nBu işlem geri alınamaz!')) {
        // AJAX ile silme işlemi
        $.ajax({
            url: 'api/delete-story.php',
            method: 'POST',
            data: { id: storyId },
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    alert('Story başarıyla silindi.');
                    window.location.href = 'story-cards.php';
                } else {
                    alert('Hata: ' + (response.message || 'Story silinirken hata oluştu.'));
                }
            },
            error: function() {
                alert('Sunucu hatası oluştu.');
            }
        });
    }
}

// Görsel önizleme fonksiyonu
function previewImage(input, previewId) {
    if (input.files && input.files[0]) {
        var reader = new FileReader();
        reader.onload = function(e) {
            $('#' + previewId).attr('src', e.target.result).show();
        }
        reader.readAsDataURL(input.files[0]);
    }
}
</script>

<?php require_once 'includes/footer.php'; ?>
