<?php
/**
 * Kategori Yönetimi Sayfası
 * Story kategorilerini yönetme
 */

session_start();
require_once __DIR__ . '/config/database.php';

$page_title = 'Kategori Yönetimi';

// Admin session kontrolü
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

// Database bağlantısı
$database = new Database();
$db = $database->getConnection();

// Form işleme
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['action'])) {
            switch ($_POST['action']) {
                case 'add_category':
                    $name = trim($_POST['name'] ?? '');
                    $display_name = trim($_POST['display_name'] ?? '');
                    $description = trim($_POST['description'] ?? '');
                    $icon_name = trim($_POST['icon_name'] ?? '');
                    $sort_order = (int)($_POST['sort_order'] ?? 0);
                    $is_active = isset($_POST['is_active']) ? 1 : 0;
                    
                    if (empty($name) || empty($display_name)) {
                        throw new Exception("Kategori adı ve görünen ad gereklidir");
                    }
                    
                    // Aynı isimde kategori var mı kontrol et
                    $stmt = $db->prepare("SELECT COUNT(*) FROM story_categories WHERE name = ?");
                    $stmt->execute([$name]);
                    if ($stmt->fetchColumn() > 0) {
                        throw new Exception("Bu isimde bir kategori zaten mevcut");
                    }
                    
                    $stmt = $db->prepare("INSERT INTO story_categories (name, display_name, description, icon_name, sort_order, is_active) VALUES (?, ?, ?, ?, ?, ?)");
                    
                    if ($stmt->execute([$name, $display_name, $description, $icon_name, $sort_order, $is_active])) {
                        $_SESSION['success_message'] = 'Kategori başarıyla eklendi';
                    } else {
                        throw new Exception("Kategori eklenirken hata oluştu");
                    }
                    break;
                    
                case 'update_category':
                    $category_id = (int)($_POST['category_id'] ?? 0);
                    $name = trim($_POST['name'] ?? '');
                    $display_name = trim($_POST['display_name'] ?? '');
                    $description = trim($_POST['description'] ?? '');
                    $icon_name = trim($_POST['icon_name'] ?? '');
                    $sort_order = (int)($_POST['sort_order'] ?? 0);
                    $is_active = isset($_POST['is_active']) ? 1 : 0;
                    
                    if ($category_id <= 0) {
                        throw new Exception("Geçersiz kategori ID");
                    }
                    
                    if (empty($name) || empty($display_name)) {
                        throw new Exception("Kategori adı ve görünen ad gereklidir");
                    }
                    
                    // Aynı isimde başka kategori var mı kontrol et
                    $stmt = $db->prepare("SELECT COUNT(*) FROM story_categories WHERE name = ? AND id != ?");
                    $stmt->execute([$name, $category_id]);
                    if ($stmt->fetchColumn() > 0) {
                        throw new Exception("Bu isimde başka bir kategori zaten mevcut");
                    }
                    
                    $stmt = $db->prepare("UPDATE story_categories SET name = ?, display_name = ?, description = ?, icon_name = ?, sort_order = ?, is_active = ? WHERE id = ?");
                    
                    if ($stmt->execute([$name, $display_name, $description, $icon_name, $sort_order, $is_active, $category_id])) {
                        $_SESSION['success_message'] = 'Kategori başarıyla güncellendi';
                    } else {
                        throw new Exception("Kategori güncellenirken hata oluştu");
                    }
                    break;
                    
                case 'delete_category':
                    $category_id = (int)($_POST['category_id'] ?? 0);
                    
                    if ($category_id <= 0) {
                        throw new Exception("Geçersiz kategori ID");
                    }
                    
                    // Bu kategoriye ait story var mı kontrol et
                    $stmt = $db->prepare("SELECT COUNT(*) FROM story_cards WHERE category_id = ?");
                    $stmt->execute([$category_id]);
                    $story_count = $stmt->fetchColumn();
                    
                    if ($story_count > 0) {
                        throw new Exception("Bu kategoriye ait $story_count adet story bulunduğu için kategori silinemez");
                    }
                    
                    $stmt = $db->prepare("DELETE FROM story_categories WHERE id = ?");
                    
                    if ($stmt->execute([$category_id])) {
                        $_SESSION['success_message'] = 'Kategori başarıyla silindi';
                    } else {
                        throw new Exception("Kategori silinirken hata oluştu");
                    }
                    break;
                    
                case 'toggle_category':
                    $category_id = (int)($_POST['category_id'] ?? 0);
                    $is_active = (int)($_POST['is_active'] ?? 0);
                    
                    if ($category_id <= 0) {
                        throw new Exception("Geçersiz kategori ID");
                    }
                    
                    $stmt = $db->prepare("UPDATE story_categories SET is_active = ? WHERE id = ?");
                    
                    if ($stmt->execute([$is_active, $category_id])) {
                        $status = $is_active ? 'aktif' : 'pasif';
                        $_SESSION['success_message'] = "Kategori başarıyla $status yapıldı";
                    } else {
                        throw new Exception("Kategori durumu değiştirilirken hata oluştu");
                    }
                    break;
            }
        }
        
        header('Location: categories.php');
        exit();
        
    } catch (Exception $e) {
        $_SESSION['error_message'] = $e->getMessage();
    }
}

// Kategorileri getir
$categories = [];
try {
    $stmt = $db->prepare("
        SELECT c.*, COUNT(s.id) as story_count 
        FROM story_categories c 
        LEFT JOIN story_cards s ON c.id = s.category_id 
        GROUP BY c.id 
        ORDER BY c.sort_order, c.name
    ");
    $stmt->execute();
    $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    error_log("Kategori listesi getirme hatası: " . $e->getMessage());
}

include 'includes/header.php';
?>

<div class="page-header">
    <h2>Kategori Yönetimi</h2>
    <p>Story kategorilerini ekleyin, düzenleyin ve yönetin</p>
</div>

<!-- Kategori Ekleme Formu -->
<div class="category-form-container">
    <div class="form-card">
        <div class="card-header">
            <h3>
                <i class="fas fa-plus-circle"></i>
                Yeni Kategori Ekle
            </h3>
        </div>
        
        <div class="card-body">
            <form method="POST" class="category-form" id="addCategoryForm">
                <input type="hidden" name="action" value="add_category">
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="name">
                            <i class="fas fa-tag"></i>
                            Kategori Adı (İngilizce) *
                        </label>
                        <input type="text" id="name" name="name" class="form-control" required 
                               placeholder="ayetler, hadisler, sozler" pattern="[a-z_]+" 
                               title="Sadece küçük harf ve alt çizgi kullanın">
                        <small>API'de kullanılacak teknik ad (sadece küçük harf ve alt çizgi)</small>
                    </div>
                    
                    <div class="form-group">
                        <label for="display_name">
                            <i class="fas fa-eye"></i>
                            Görünen Ad *
                        </label>
                        <input type="text" id="display_name" name="display_name" class="form-control" required 
                               placeholder="Ayetler, Hadisler, Sözler">
                        <small>Kullanıcılara gösterilecek ad</small>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="description">
                        <i class="fas fa-align-left"></i>
                        Açıklama
                    </label>
                    <textarea id="description" name="description" class="form-control" rows="3" 
                              placeholder="Kategori açıklaması (isteğe bağlı)"></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="icon_name">
                            <i class="fas fa-icons"></i>
                            İkon Adı
                        </label>
                        <select id="icon_name" name="icon_name" class="form-control">
                            <option value="">İkon Seçin</option>
                            <option value="ic_quran">📖 Kuran</option>
                            <option value="ic_hadith">📜 Hadis</option>
                            <option value="ic_quote">💬 Söz</option>
                            <option value="ic_prayer">🤲 Dua</option>
                            <option value="ic_mosque">🕌 Cami</option>
                            <option value="ic_star">⭐ Yıldız</option>
                            <option value="ic_heart">❤️ Kalp</option>
                            <option value="ic_book">📚 Kitap</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="sort_order">
                            <i class="fas fa-sort-numeric-up"></i>
                            Sıralama
                        </label>
                        <input type="number" id="sort_order" name="sort_order" class="form-control" 
                               value="0" min="0" max="999">
                        <small>Küçük sayılar önce gösterilir</small>
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" name="is_active" value="1" checked>
                        <i class="fas fa-check-circle"></i>
                        Kategoriyi aktif olarak yayınla
                    </label>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-save"></i>
                        Kategori Ekle
                    </button>
                    <button type="reset" class="btn btn-secondary">
                        <i class="fas fa-undo"></i>
                        Temizle
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Kategori Listesi -->
<div class="categories-list">
    <div class="card">
        <div class="card-header">
            <h3>
                <i class="fas fa-list"></i>
                Mevcut Kategoriler (<?php echo count($categories); ?> adet)
            </h3>
        </div>
        
        <div class="card-body">
            <?php if (!empty($categories)): ?>
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Sıra</th>
                                <th>Kategori</th>
                                <th>İkon</th>
                                <th>Story Sayısı</th>
                                <th>Durum</th>
                                <th>Oluşturulma</th>
                                <th>İşlemler</th>
                            </tr>
                        </thead>
                        <tbody id="categoriesTable">
                            <?php foreach ($categories as $category): ?>
                                <tr data-id="<?php echo $category['id']; ?>">
                                    <td>
                                        <span class="sort-handle">
                                            <i class="fas fa-grip-vertical"></i>
                                        </span>
                                        <strong><?php echo $category['sort_order']; ?></strong>
                                    </td>
                                    <td>
                                        <div class="category-info">
                                            <div class="category-display-name">
                                                <?php echo htmlspecialchars($category['display_name']); ?>
                                            </div>
                                            <div class="category-name">
                                                <code><?php echo htmlspecialchars($category['name']); ?></code>
                                            </div>
                                            <?php if (!empty($category['description'])): ?>
                                                <div class="category-description">
                                                    <?php echo htmlspecialchars($category['description']); ?>
                                                </div>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                    <td>
                                        <?php if (!empty($category['icon_name'])): ?>
                                            <span class="category-icon" title="<?php echo htmlspecialchars($category['icon_name']); ?>">
                                                <?php
                                                $icons = [
                                                    'ic_quran' => '📖',
                                                    'ic_hadith' => '📜',
                                                    'ic_quote' => '💬',
                                                    'ic_prayer' => '🤲',
                                                    'ic_mosque' => '🕌',
                                                    'ic_star' => '⭐',
                                                    'ic_heart' => '❤️',
                                                    'ic_book' => '📚'
                                                ];
                                                echo $icons[$category['icon_name']] ?? '📁';
                                                ?>
                                            </span>
                                        <?php else: ?>
                                            <span class="no-icon">-</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="story-count <?php echo $category['story_count'] > 0 ? 'has-stories' : 'no-stories'; ?>">
                                            <?php echo number_format($category['story_count']); ?>
                                        </span>
                                    </td>
                                    <td>
                                        <?php if ($category['is_active']): ?>
                                            <span class="status-badge active">Aktif</span>
                                        <?php else: ?>
                                            <span class="status-badge inactive">Pasif</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="date-info">
                                            <?php echo date('d.m.Y', strtotime($category['created_at'])); ?>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <button type="button" class="btn btn-sm btn-info" 
                                                    onclick="editCategory(<?php echo htmlspecialchars(json_encode($category)); ?>)">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            
                                            <form method="POST" style="display: inline;">
                                                <input type="hidden" name="action" value="toggle_category">
                                                <input type="hidden" name="category_id" value="<?php echo $category['id']; ?>">
                                                <input type="hidden" name="is_active" value="<?php echo $category['is_active'] ? 0 : 1; ?>">
                                                <button type="submit" class="btn btn-sm <?php echo $category['is_active'] ? 'btn-warning' : 'btn-success'; ?>" 
                                                        title="<?php echo $category['is_active'] ? 'Pasif Yap' : 'Aktif Yap'; ?>">
                                                    <i class="fas <?php echo $category['is_active'] ? 'fa-pause' : 'fa-play'; ?>"></i>
                                                </button>
                                            </form>
                                            
                                            <?php if ($category['story_count'] == 0): ?>
                                                <form method="POST" style="display: inline;" 
                                                      onsubmit="return confirm('Bu kategoriyi silmek istediğinizden emin misiniz?')">
                                                    <input type="hidden" name="action" value="delete_category">
                                                    <input type="hidden" name="category_id" value="<?php echo $category['id']; ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger" title="Sil">
                                                        <i class="fas fa-trash"></i>
                                                    </button>
                                                </form>
                                            <?php else: ?>
                                                <button type="button" class="btn btn-sm btn-secondary" disabled 
                                                        title="Bu kategoriye ait story'ler olduğu için silinemez">
                                                    <i class="fas fa-lock"></i>
                                                </button>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div class="empty-state">
                    <i class="fas fa-tags"></i>
                    <h3>Henüz kategori eklenmemiş</h3>
                    <p>İlk kategorinizi eklemek için yukarıdaki formu kullanın.</p>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Kategori Düzenleme Modal -->
<div id="editCategoryModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Kategori Düzenle</h3>
            <button type="button" class="modal-close" onclick="closeEditModal()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <form method="POST" id="editCategoryForm">
            <input type="hidden" name="action" value="update_category">
            <input type="hidden" name="category_id" id="edit_category_id">
            
            <div class="modal-body">
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_name">Kategori Adı (İngilizce) *</label>
                        <input type="text" id="edit_name" name="name" class="form-control" required 
                               pattern="[a-z_]+" title="Sadece küçük harf ve alt çizgi kullanın">
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_display_name">Görünen Ad *</label>
                        <input type="text" id="edit_display_name" name="display_name" class="form-control" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="edit_description">Açıklama</label>
                    <textarea id="edit_description" name="description" class="form-control" rows="3"></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="edit_icon_name">İkon Adı</label>
                        <select id="edit_icon_name" name="icon_name" class="form-control">
                            <option value="">İkon Seçin</option>
                            <option value="ic_quran">📖 Kuran</option>
                            <option value="ic_hadith">📜 Hadis</option>
                            <option value="ic_quote">💬 Söz</option>
                            <option value="ic_prayer">🤲 Dua</option>
                            <option value="ic_mosque">🕌 Cami</option>
                            <option value="ic_star">⭐ Yıldız</option>
                            <option value="ic_heart">❤️ Kalp</option>
                            <option value="ic_book">📚 Kitap</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="edit_sort_order">Sıralama</label>
                        <input type="number" id="edit_sort_order" name="sort_order" class="form-control" 
                               min="0" max="999">
                    </div>
                </div>
                
                <div class="form-group">
                    <label class="checkbox-label">
                        <input type="checkbox" id="edit_is_active" name="is_active" value="1">
                        <i class="fas fa-check-circle"></i>
                        Kategoriyi aktif olarak yayınla
                    </label>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i>
                    Güncelle
                </button>
                <button type="button" class="btn btn-secondary" onclick="closeEditModal()">
                    İptal
                </button>
            </div>
        </form>
    </div>
</div>

<script>
// Kategori düzenleme modalını aç
function editCategory(category) {
    document.getElementById('edit_category_id').value = category.id;
    document.getElementById('edit_name').value = category.name;
    document.getElementById('edit_display_name').value = category.display_name;
    document.getElementById('edit_description').value = category.description || '';
    document.getElementById('edit_icon_name').value = category.icon_name || '';
    document.getElementById('edit_sort_order').value = category.sort_order;
    document.getElementById('edit_is_active').checked = category.is_active == 1;
    
    document.getElementById('editCategoryModal').style.display = 'flex';
}

// Modal kapatma
function closeEditModal() {
    document.getElementById('editCategoryModal').style.display = 'none';
}

// Modal dışına tıklayınca kapatma
window.onclick = function(event) {
    const modal = document.getElementById('editCategoryModal');
    if (event.target === modal) {
        closeEditModal();
    }
}

// Form validasyonu
document.getElementById('addCategoryForm').addEventListener('submit', function(e) {
    const name = document.getElementById('name').value;
    const displayName = document.getElementById('display_name').value;
    
    if (!name.match(/^[a-z_]+$/)) {
        e.preventDefault();
        alert('Kategori adı sadece küçük harf ve alt çizgi içerebilir!');
        return false;
    }
    
    if (name.length < 2) {
        e.preventDefault();
        alert('Kategori adı en az 2 karakter olmalıdır!');
        return false;
    }
    
    if (displayName.length < 2) {
        e.preventDefault();
        alert('Görünen ad en az 2 karakter olmalıdır!');
        return false;
    }
});

document.getElementById('editCategoryForm').addEventListener('submit', function(e) {
    const name = document.getElementById('edit_name').value;
    const displayName = document.getElementById('edit_display_name').value;
    
    if (!name.match(/^[a-z_]+$/)) {
        e.preventDefault();
        alert('Kategori adı sadece küçük harf ve alt çizgi içerebilir!');
        return false;
    }
    
    if (name.length < 2) {
        e.preventDefault();
        alert('Kategori adı en az 2 karakter olmalıdır!');
        return false;
    }
    
    if (displayName.length < 2) {
        e.preventDefault();
        alert('Görünen ad en az 2 karakter olmalıdır!');
        return false;
    }
});
</script>

<style>
.category-form-container {
    margin-bottom: 2rem;
}

.form-card, .card {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    overflow: hidden;
}

.card-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 1.5rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.card-header h3 {
    margin: 0;
    font-size: 1.25rem;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.card-body {
    padding: 2rem;
}

.category-form {
    max-width: 800px;
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

.form-group {
    margin-bottom: 1.5rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #374151;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.form-group input, .form-group select, .form-group textarea {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #e5e7eb;
    border-radius: 8px;
    font-size: 0.9rem;
    transition: border-color 0.3s;
}

.form-group input:focus, .form-group select:focus, .form-group textarea:focus {
    outline: none;
    border-color: #4f46e5;
}

.form-group small {
    color: #6b7280;
    font-size: 0.8rem;
    margin-top: 0.25rem;
    display: block;
}

.checkbox-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
    padding: 0.75rem;
    background: #f9fafb;
    border-radius: 8px;
    border: 2px solid #e5e7eb;
    transition: all 0.3s;
}

.checkbox-label:hover {
    background: #f3f4f6;
    border-color: #d1d5db;
}

.checkbox-label input[type="checkbox"] {
    width: auto;
    margin: 0;
}

.form-actions {
    display: flex;
    gap: 1rem;
    margin-top: 2rem;
}

.btn {
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s;
    text-decoration: none;
}

.btn-sm {
    padding: 0.5rem 1rem;
    font-size: 0.9rem;
}

.btn-primary {
    background: linear-gradient(135deg, #4f46e5, #7c3aed);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(79, 70, 229, 0.4);
}

.btn-secondary {
    background: #f3f4f6;
    color: #374151;
}

.btn-secondary:hover {
    background: #e5e7eb;
}

.btn-info {
    background: #06b6d4;
    color: white;
}

.btn-success {
    background: #10b981;
    color: white;
}

.btn-warning {
    background: #f59e0b;
    color: white;
}

.btn-danger {
    background: #ef4444;
    color: white;
}

.categories-list {
    margin-top: 2rem;
}

.table-responsive {
    overflow-x: auto;
}

.table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 1rem;
}

.table th, .table td {
    padding: 1rem;
    text-align: left;
    border-bottom: 1px solid #e5e7eb;
}

.table th {
    background: #f9fafb;
    font-weight: 600;
    color: #374151;
}

.sort-handle {
    cursor: grab;
    color: #9ca3af;
    margin-right: 0.5rem;
}

.sort-handle:active {
    cursor: grabbing;
}

.category-info {
    max-width: 250px;
}

.category-display-name {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.25rem;
}

.category-name {
    margin-bottom: 0.25rem;
}

.category-name code {
    background: #f3f4f6;
    color: #6b7280;
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
    font-size: 0.8rem;
}

.category-description {
    color: #6b7280;
    font-size: 0.9rem;
    line-height: 1.4;
}

.category-icon {
    font-size: 1.5rem;
}

.no-icon {
    color: #9ca3af;
    font-style: italic;
}

.story-count {
    font-weight: 600;
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
    font-size: 0.9rem;
}

.story-count.has-stories {
    background: #dcfce7;
    color: #166534;
}

.story-count.no-stories {
    background: #f3f4f6;
    color: #6b7280;
}

.status-badge {
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-weight: 600;
    font-size: 0.8rem;
    text-transform: uppercase;
}

.status-badge.active {
    background: #dcfce7;
    color: #166534;
}

.status-badge.inactive {
    background: #fee2e2;
    color: #991b1b;
}

.action-buttons {
    display: flex;
    gap: 0.5rem;
    align-items: center;
}

.empty-state {
    text-align: center;
    padding: 3rem;
    color: #6b7280;
}

.empty-state i {
    font-size: 3rem;
    margin-bottom: 1rem;
    color: #d1d5db;
}

.empty-state h3 {
    margin-bottom: 0.5rem;
    color: #374151;
}

/* Modal Styles */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}

.modal-content {
    background: white;
    border-radius: 12px;
    width: 90%;
    max-width: 600px;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1.5rem;
    border-bottom: 1px solid #e5e7eb;
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 12px 12px 0 0;
}

.modal-header h3 {
    margin: 0;
    font-size: 1.25rem;
}

.modal-close {
    background: none;
    border: none;
    font-size: 1.5rem;
    color: white;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 50%;
    transition: all 0.3s;
}

.modal-close:hover {
    background: rgba(255, 255, 255, 0.2);
}

.modal-body {
    padding: 2rem;
}

.modal-footer {
    display: flex;
    justify-content: flex-end;
    gap: 1rem;
    padding: 1.5rem;
    border-top: 1px solid #e5e7eb;
    background: #f9fafb;
    border-radius: 0 0 12px 12px;
}

@media (max-width: 768px) {
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .category-form {
        max-width: none;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 0.25rem;
    }
    
    .modal-content {
        width: 95%;
        margin: 1rem;
    }
    
    .modal-footer {
        flex-direction: column;
    }
}
</style>

<?php include 'includes/footer.php'; ?>
