<?php
/**
 * Firebase Cloud Messaging Service
 * Firebase FCM HTTP v1 API kullanarak bildirim gönderme
 */
class FirebaseService {
    private $db;
    private $config;
    private $accessToken;
    private $tokenExpiry;
    
    public function __construct($database) {
        $this->db = $database;
        $this->loadConfig();
    }
    
    /**
     * Firebase yapılandırmasını yükle
     */
    private function loadConfig() {
        try {
            $stmt = $this->db->prepare("SELECT project_id, service_account_json FROM firebase_config WHERE is_active = 1 LIMIT 1");
            $stmt->execute();
            $configRow = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$configRow) {
                throw new Exception("Firebase yapılandırması bulunamadı. Lütfen Firebase ayarlarını yapılandırın.");
            }
            
            // Service Account JSON'ı parse et
            $serviceAccount = json_decode($configRow['service_account_json'], true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("Firebase Service Account JSON formatı geçersiz.");
            }
            
            // Config array'ini oluştur
            $this->config = [
                'project_id' => $configRow['project_id'],
                'private_key' => $serviceAccount['private_key'],
                'client_email' => $serviceAccount['client_email'],
                'service_account' => $serviceAccount
            ];
            
        } catch (Exception $e) {
            error_log("Firebase config yükleme hatası: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Firebase yapılandırmasının geçerli olup olmadığını kontrol et
     */
    public function isConfigured() {
        return !empty($this->config) && 
               !empty($this->config['project_id']) && 
               !empty($this->config['private_key']) && 
               !empty($this->config['client_email']);
    }
    
    /**
     * JWT token oluştur
     */
    private function createJWT() {
        try {
            $header = json_encode(['typ' => 'JWT', 'alg' => 'RS256']);
            $now = time();
            $payload = json_encode([
                'iss' => $this->config['client_email'],
                'scope' => 'https://www.googleapis.com/auth/firebase.messaging',
                'aud' => 'https://oauth2.googleapis.com/token',
                'exp' => $now + 3600,
                'iat' => $now
            ]);
            
            $base64Header = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($header));
            $base64Payload = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($payload));
            
            // Private key'i kontrol et
            $privateKey = openssl_pkey_get_private($this->config['private_key']);
            if ($privateKey === false) {
                $opensslError = '';
                while ($msg = openssl_error_string()) {
                    $opensslError .= $msg . '; ';
                }
                throw new Exception("Private key geçersiz: " . $opensslError);
            }
            
            $signature = '';
            $signResult = openssl_sign($base64Header . "." . $base64Payload, $signature, $privateKey, OPENSSL_ALGO_SHA256);
            
            if (!$signResult) {
                $opensslError = '';
                while ($msg = openssl_error_string()) {
                    $opensslError .= $msg . '; ';
                }
                openssl_free_key($privateKey);
                throw new Exception("JWT imzalama başarısız: " . $opensslError);
            }
            
            openssl_free_key($privateKey);
            
            $base64Signature = str_replace(['+', '/', '='], ['-', '_', ''], base64_encode($signature));
            
            return $base64Header . "." . $base64Payload . "." . $base64Signature;
            
        } catch (Exception $e) {
            error_log("JWT oluşturma hatası: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Access token al
     */
    private function getAccessToken() {
        try {
            $jwt = $this->createJWT();
            
            $postData = http_build_query([
                'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
                'assertion' => $jwt
            ]);
            
            // cURL kullan
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, 'https://oauth2.googleapis.com/token');
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/x-www-form-urlencoded'
            ]);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
            
            if ($response === false || !empty($curlError)) {
                throw new Exception("OAuth2 token isteği başarısız: " . $curlError);
            }
            
            if ($httpCode >= 400) {
                throw new Exception("OAuth2 token isteği HTTP $httpCode hatası: " . $response);
            }
            
            $data = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("OAuth2 response JSON parse hatası: " . json_last_error_msg() . " - Response: " . $response);
            }
            
            if (!isset($data['access_token'])) {
                $errorMsg = "Access token alınamadı";
                if (isset($data['error'])) {
                    $errorMsg .= " - " . $data['error'];
                    if (isset($data['error_description'])) {
                        $errorMsg .= ": " . $data['error_description'];
                    }
                }
                $errorMsg .= " - Full response: " . $response;
                throw new Exception($errorMsg);
            }
            
            return $data['access_token'];
            
        } catch (Exception $e) {
            error_log("Firebase Access Token Hatası: " . $e->getMessage());
            throw $e;
        }
    }
    
    /**
     * Tüm kullanıcılara bildirim gönder
     */
    public function sendToAll($title, $body, $icon = 'ic_notification', $sound = 'default', $clickAction = null) {
        return $this->sendToTopic('all_users', $title, $body, $icon, $sound, $clickAction);
    }
    
    /**
     * Belirli topic'e bildirim gönder
     */
    public function sendToTopic($topic, $title, $body, $icon = 'ic_notification', $sound = 'default', $clickAction = null) {
        try {
            $accessToken = $this->getAccessToken();
            
            $message = [
                'message' => [
                    'topic' => $topic,
                    'notification' => [
                        'title' => $title,
                        'body' => $body
                    ],
                    'data' => [
                        'icon' => $icon,
                        'sound' => $sound
                    ],
                    'android' => [
                        'notification' => [
                            'icon' => $icon,
                            'sound' => $sound,
                            'click_action' => $clickAction ?: 'FLUTTER_NOTIFICATION_CLICK'
                        ]
                    ]
                ]
            ];
            
            $url = 'https://fcm.googleapis.com/v1/projects/' . $this->config['project_id'] . '/messages:send';
            
            // cURL kullan
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken,
                'Content-Type: application/json'
            ]);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
            
            if ($response === false || !empty($curlError)) {
                throw new Exception("FCM isteği başarısız: " . $curlError);
            }
            
            $responseData = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("FCM response JSON parse hatası: " . json_last_error_msg() . " - Response: " . $response);
            }
            
            // HTTP status code kontrolü
            if ($httpCode >= 400) {
                $errorMessage = isset($responseData['error']['message']) 
                    ? $responseData['error']['message'] 
                    : "HTTP $httpCode hatası";
                throw new Exception($errorMessage . " - Response: " . $response);
            }
            
            // FCM response kontrolü
            if (isset($responseData['error'])) {
                throw new Exception("FCM hatası: " . $responseData['error']['message']);
            }
            
            return [
                'success' => true,
                'response' => $responseData,
                'message' => 'Bildirim başarıyla gönderildi',
                'message_id' => $responseData['name'] ?? null
            ];
            
        } catch (Exception $e) {
            error_log("Firebase bildirim hatası: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'Bildirim gönderilemedi'
            ];
        }
    }
    
    /**
     * Belirli token'a bildirim gönder
     */
    public function sendToToken($token, $title, $body, $icon = 'ic_notification', $sound = 'default', $clickAction = null) {
        try {
            $accessToken = $this->getAccessToken();
            
            $message = [
                'message' => [
                    'token' => $token,
                    'notification' => [
                        'title' => $title,
                        'body' => $body
                    ],
                    'data' => [
                        'icon' => $icon,
                        'sound' => $sound
                    ],
                    'android' => [
                        'notification' => [
                            'icon' => $icon,
                            'sound' => $sound,
                            'click_action' => $clickAction ?: 'FLUTTER_NOTIFICATION_CLICK'
                        ]
                    ]
                ]
            ];
            
            $url = 'https://fcm.googleapis.com/v1/projects/' . $this->config['project_id'] . '/messages:send';
            
            // cURL kullan
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($message));
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Authorization: Bearer ' . $accessToken,
                'Content-Type: application/json'
            ]);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curlError = curl_error($ch);
            curl_close($ch);
            
            if ($response === false || !empty($curlError)) {
                throw new Exception("FCM isteği başarısız: " . $curlError);
            }
            
            $responseData = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("FCM response JSON parse hatası: " . json_last_error_msg() . " - Response: " . $response);
            }
            
            // HTTP status code kontrolü
            if ($httpCode >= 400) {
                $errorMessage = isset($responseData['error']['message']) 
                    ? $responseData['error']['message'] 
                    : "HTTP $httpCode hatası";
                throw new Exception($errorMessage . " - Response: " . $response);
            }
            
            // FCM response kontrolü
            if (isset($responseData['error'])) {
                throw new Exception("FCM hatası: " . $responseData['error']['message']);
            }
            
            return [
                'success' => true,
                'response' => $responseData,
                'message' => 'Bildirim başarıyla gönderildi',
                'message_id' => $responseData['name'] ?? null
            ];
            
        } catch (Exception $e) {
            error_log("Firebase bildirim hatası: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'message' => 'Bildirim gönderilemedi'
            ];
        }
    }
    
    /**
     * Bildirim geçmişini kaydet
     */
    public function saveNotificationHistory($templateId, $title, $body, $icon, $sound, $clickAction, $targetType, $targetValue, $result, $sentBy) {
        try {
            // Debug log
            error_log("saveNotificationHistory çağrıldı - sentBy: " . var_export($sentBy, true));
            
            // sentBy kontrolü
            if (empty($sentBy)) {
                error_log("UYARI: sentBy değeri boş! Session admin_id kontrol edilmeli.");
                $sentBy = 1; // Varsayılan admin ID
            }
            
            $stmt = $this->db->prepare("
                INSERT INTO sent_notifications 
                (template_id, title, body, icon, sound, click_action, target_type, target_value, status, response_data, sent_count, failed_count, sent_by) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");
            
            $status = $result['success'] ? 'sent' : 'failed';
            $sentCount = $result['success'] ? 1 : 0;
            $failedCount = $result['success'] ? 0 : 1;
            
            $params = [
                $templateId,
                $title,
                $body,
                $icon,
                $sound,
                $clickAction,
                $targetType,
                $targetValue,
                $status,
                json_encode($result),
                $sentCount,
                $failedCount,
                $sentBy
            ];
            
            error_log("SQL parametreleri: " . json_encode($params));
            
            $stmt->execute($params);
            
            $insertId = $this->db->lastInsertId();
            error_log("Bildirim geçmişi başarıyla kaydedildi. ID: " . $insertId);
            
            return $insertId;
            
        } catch (Exception $e) {
            error_log("Bildirim geçmişi kaydetme hatası: " . $e->getMessage());
            error_log("SQL Hatası: " . print_r($this->db->errorInfo(), true));
            return false;
        }
    }
}
