<?php
/**
 * Bildirim Gönderme Sayfası
 * Firebase FCM ile push notification gönderme
 */

session_start();
require_once 'config/database.php';
require_once 'includes/FirebaseService.php';

$page_title = 'Bildirim Gönder';

// Admin session kontrolü
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

// Debug: Session bilgilerini logla
error_log("Admin session - ID: " . ($_SESSION['admin_id'] ?? 'YOK') . ", Username: " . ($_SESSION['username'] ?? 'YOK'));

// Database bağlantısı
$database = new Database();
$db = $database->getConnection();

// Firebase service
try {
    $firebaseService = new FirebaseService($db);
    $isFirebaseConfigured = $firebaseService->isConfigured();
} catch (Exception $e) {
    $isFirebaseConfigured = false;
    $firebaseError = $e->getMessage();
}

// Form işleme
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!$isFirebaseConfigured) {
        $_SESSION['error_message'] = 'Firebase yapılandırması eksik. Lütfen Firebase ayarlarını yapılandırın.';
        header('Location: notifications.php');
        exit();
    }
    try {
        $title = trim($_POST['title'] ?? '');
        $body = trim($_POST['body'] ?? '');
        $icon = trim($_POST['icon'] ?? 'ic_notification');
        $sound = trim($_POST['sound'] ?? 'default');
        $clickAction = trim($_POST['click_action'] ?? '');
        $targetType = $_POST['target_type'] ?? 'all';
        $targetValue = trim($_POST['target_value'] ?? '');
        $templateId = !empty($_POST['template_id']) ? (int)$_POST['template_id'] : null;
        
        // Validasyon
        if (empty($title) || empty($body)) {
            throw new Exception("Başlık ve mesaj alanları zorunludur");
        }
        
        // Bildirim gönder
        $result = null;
        switch ($targetType) {
            case 'all':
                $result = $firebaseService->sendToAll($title, $body, $icon, $sound, $clickAction);
                break;
            case 'topic':
                if (empty($targetValue)) {
                    throw new Exception("Topic adı gereklidir");
                }
                $result = $firebaseService->sendToTopic($targetValue, $title, $body, $icon, $sound, $clickAction);
                break;
            case 'token':
                if (empty($targetValue)) {
                    throw new Exception("Device token gereklidir");
                }
                $result = $firebaseService->sendToToken($targetValue, $title, $body, $icon, $sound, $clickAction);
                break;
            default:
                throw new Exception("Geçersiz hedef türü");
        }
        
        // Geçmişe kaydet
        $firebaseService->saveNotificationHistory(
            $templateId,
            $title,
            $body,
            $icon,
            $sound,
            $clickAction,
            $targetType,
            $targetValue,
            $result,
            $_SESSION['admin_id']
        );
        
        if ($result['success']) {
            $_SESSION['success_message'] = $result['message'];
        } else {
            $_SESSION['error_message'] = $result['message'];
        }
        
        header('Location: notifications.php');
        exit();
        
    } catch (Exception $e) {
        $_SESSION['error_message'] = $e->getMessage();
    }
}

// Bildirim şablonlarını getir
$templates = [];
try {
    $stmt = $db->prepare("SELECT * FROM notification_templates WHERE is_active = 1 ORDER BY name");
    $stmt->execute();
    $templates = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    error_log("Template getirme hatası: " . $e->getMessage());
}

include 'includes/header.php';
?>

<div class="page-header">
    <h2>Bildirim Gönder</h2>
    <p>Mobil uygulama kullanıcılarına push notification gönderin</p>
    
    <!-- Firebase Ayarları Butonu -->
    <div class="header-actions">
        <button type="button" class="btn btn-secondary" onclick="toggleFirebaseSettings()">
            <i class="fas fa-cog"></i>
            Firebase Ayarları
        </button>
    </div>
</div>

<!-- Firebase Ayarları Modal -->
<div id="firebaseSettingsModal" class="modal" style="display: none;">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Firebase HTTP v1 API Ayarları</h3>
            <button type="button" class="modal-close" onclick="toggleFirebaseSettings()">
                <i class="fas fa-times"></i>
            </button>
        </div>
        
        <form id="firebaseSettingsForm" method="POST" action="firebase-settings.php">
            <div class="modal-body">
                <div class="form-group">
                    <label for="project_id">Project ID *</label>
                    <input type="text" id="project_id" name="project_id" required 
                           placeholder="Firebase Project ID'nizi girin">
                    <small>Firebase Console > Project Settings > General sekmesinde bulabilirsiniz</small>
                </div>
                
                <div class="form-group">
                    <label for="service_account_json">Service Account JSON *</label>
                    <textarea id="service_account_json" name="service_account_json" required rows="10" 
                              placeholder="Service Account JSON içeriğini buraya yapıştırın"></textarea>
                    <small>Firebase Console > Project Settings > Service Accounts > Generate new private key</small>
                </div>
                
                <div class="firebase-help">
                    <h4>Firebase HTTP v1 API Kurulum Rehberi:</h4>
                    <ol>
                        <li>Firebase Console'a gidin: <a href="https://console.firebase.google.com" target="_blank">console.firebase.google.com</a></li>
                        <li>Projenizi seçin</li>
                        <li>Project Settings > General sekmesinden Project ID'yi kopyalayın</li>
                        <li>Service Accounts sekmesine gidin</li>
                        <li>"Generate new private key" butonuna tıklayın</li>
                        <li>İndirilen JSON dosyasının içeriğini yukarıdaki alana yapıştırın</li>
                    </ol>
                </div>
            </div>
            
            <div class="modal-footer">
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-save"></i>
                    Ayarları Kaydet
                </button>
                <button type="button" class="btn btn-secondary" onclick="toggleFirebaseSettings()">
                    İptal
                </button>
            </div>
        </form>
    </div>
</div>

<?php if (!$isFirebaseConfigured): ?>
<!-- Firebase Yapılandırma Uyarısı -->
<div class="config-warning">
    <div class="warning-content">
        <div class="warning-icon">
            <i class="fas fa-exclamation-triangle"></i>
        </div>
        <div class="warning-text">
            <h3>Firebase Yapılandırması Gerekli</h3>
            <p><?php echo isset($firebaseError) ? htmlspecialchars($firebaseError) : 'Firebase HTTP v1 API ayarları yapılandırılmamış. Bildirim gönderebilmek için Firebase ayarlarını yapılandırmanız gerekiyor.'; ?></p>
        </div>
        <div class="warning-action">
            <button type="button" class="btn btn-primary" onclick="toggleFirebaseSettings()">
                <i class="fas fa-cog"></i>
                Firebase Ayarlarını Yapılandır
            </button>
        </div>
    </div>
</div>
<?php endif; ?>

<div class="notification-container">
    <div class="notification-form-section">
        <div class="form-card">
            <h3>Yeni Bildirim Gönder</h3>
            
            <?php if (!$isFirebaseConfigured): ?>
            <div class="form-disabled-overlay">
                <p>Firebase yapılandırması tamamlanana kadar bildirim gönderemezsiniz.</p>
            </div>
            <?php endif; ?>
            
            <form method="POST" class="notification-form" <?php echo !$isFirebaseConfigured ? 'style="pointer-events: none; opacity: 0.6;"' : ''; ?>>
                <!-- Şablon Seçimi -->
                <div class="form-group">
                    <label for="template_select">Şablon Seç (Opsiyonel)</label>
                    <select id="template_select" onchange="loadTemplate()">
                        <option value="">Özel bildirim yaz</option>
                        <?php foreach ($templates as $template): ?>
                            <option value="<?php echo $template['id']; ?>" 
                                    data-title="<?php echo htmlspecialchars($template['title']); ?>"
                                    data-body="<?php echo htmlspecialchars($template['body']); ?>"
                                    data-icon="<?php echo htmlspecialchars($template['icon']); ?>"
                                    data-sound="<?php echo htmlspecialchars($template['sound']); ?>"
                                    data-click-action="<?php echo htmlspecialchars($template['click_action']); ?>">
                                <?php echo htmlspecialchars($template['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Bildirim İçeriği -->
                <div class="form-row">
                    <div class="form-group">
                        <label for="title">Başlık *</label>
                        <input type="text" id="title" name="title" required maxlength="200" 
                               placeholder="Bildirim başlığı">
                    </div>
                    
                    <div class="form-group">
                        <label for="icon">İkon</label>
                        <select id="icon" name="icon">
                            <option value="ic_notification">Varsayılan</option>
                            <option value="ic_prayer">Namaz</option>
                            <option value="ic_quran">Kuran</option>
                            <option value="ic_hadith">Hadis</option>
                            <option value="ic_quote">Söz</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="body">Mesaj *</label>
                    <textarea id="body" name="body" required rows="4" maxlength="500" 
                              placeholder="Bildirim mesajı"></textarea>
                    <small class="char-count">0/500 karakter</small>
                </div>
                
                <!-- Gelişmiş Ayarlar -->
                <div class="form-row">
                    <div class="form-group">
                        <label for="sound">Ses</label>
                        <select id="sound" name="sound">
                            <option value="default">Varsayılan</option>
                            <option value="ezan">Ezan Sesi</option>
                            <option value="notification">Bildirim Sesi</option>
                            <option value="silent">Sessiz</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label for="click_action">Tıklama Eylemi</label>
                        <input type="text" id="click_action" name="click_action" 
                               placeholder="Örn: OPEN_STORY_DETAIL">
                    </div>
                </div>
                
                <!-- Hedef Seçimi -->
                <div class="form-group">
                    <label>Hedef Kitle</label>
                    <div class="radio-group">
                        <label class="radio-option">
                            <input type="radio" name="target_type" value="all" checked onchange="toggleTargetValue()">
                            <span>Tüm Kullanıcılar</span>
                        </label>
                        <label class="radio-option">
                            <input type="radio" name="target_type" value="topic" onchange="toggleTargetValue()">
                            <span>Belirli Topic</span>
                        </label>
                        <label class="radio-option">
                            <input type="radio" name="target_type" value="token" onchange="toggleTargetValue()">
                            <span>Belirli Cihaz</span>
                        </label>
                    </div>
                </div>
                
                <div class="form-group" id="target_value_group" style="display: none;">
                    <label for="target_value">Hedef Değer</label>
                    <input type="text" id="target_value" name="target_value" 
                           placeholder="Topic adı veya device token">
                    <small>Topic için: all_users, android_users vb. Token için: FCM device token</small>
                </div>
                
                <input type="hidden" id="template_id" name="template_id">
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-paper-plane"></i>
                        Bildirim Gönder
                    </button>
                    <button type="button" class="btn btn-secondary" onclick="resetForm()">
                        <i class="fas fa-undo"></i>
                        Temizle
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Önizleme -->
    <div class="notification-preview-section">
        <div class="preview-card">
            <h3>Bildirim Önizlemesi</h3>
            <div class="phone-mockup">
                <div class="notification-preview">
                    <div class="notification-icon">
                        <i class="fas fa-bell"></i>
                    </div>
                    <div class="notification-content">
                        <div class="notification-title" id="preview-title">Bildirim Başlığı</div>
                        <div class="notification-body" id="preview-body">Bildirim mesajı burada görünecek</div>
                        <div class="notification-time">Şimdi</div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Son Gönderilen Bildirimler -->
        <div class="recent-notifications">
            <h3>Son Gönderilen Bildirimler</h3>
            <?php
            try {
                $stmt = $db->prepare("
                    SELECT sn.*, au.username 
                    FROM sent_notifications sn 
                    LEFT JOIN admin_users au ON sn.sent_by = au.id 
                    ORDER BY sn.sent_at DESC 
                    LIMIT 5
                ");
                $stmt->execute();
                $recentNotifications = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                if ($recentNotifications):
            ?>
                <div class="recent-list">
                    <?php foreach ($recentNotifications as $notification): ?>
                        <div class="recent-item <?php echo $notification['status']; ?>">
                            <div class="recent-content">
                                <div class="recent-title"><?php echo htmlspecialchars($notification['title']); ?></div>
                                <div class="recent-body"><?php echo htmlspecialchars(substr($notification['body'], 0, 50)) . '...'; ?></div>
                                <div class="recent-meta">
                                    <span class="status-badge status-<?php echo $notification['status']; ?>">
                                        <?php echo ucfirst($notification['status']); ?>
                                    </span>
                                    <span class="sent-by">by <?php echo htmlspecialchars($notification['username']); ?></span>
                                    <span class="sent-time"><?php echo date('d.m.Y H:i', strtotime($notification['sent_at'])); ?></span>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php else: ?>
                <p class="no-data">Henüz bildirim gönderilmemiş</p>
            <?php endif; ?>
            <?php } catch (Exception $e) { ?>
                <p class="error">Bildirim geçmişi yüklenemedi</p>
            <?php } ?>
        </div>
    </div>
</div>

<script>
// Şablon yükleme
function loadTemplate() {
    const select = document.getElementById('template_select');
    const option = select.options[select.selectedIndex];
    
    if (option.value) {
        document.getElementById('title').value = option.dataset.title || '';
        document.getElementById('body').value = option.dataset.body || '';
        document.getElementById('icon').value = option.dataset.icon || 'ic_notification';
        document.getElementById('sound').value = option.dataset.sound || 'default';
        document.getElementById('click_action').value = option.dataset.clickAction || '';
        document.getElementById('template_id').value = option.value;
        
        updatePreview();
        updateCharCount();
    }
}

// Hedef değer alanını göster/gizle
function toggleTargetValue() {
    const targetType = document.querySelector('input[name="target_type"]:checked').value;
    const targetValueGroup = document.getElementById('target_value_group');
    const targetValueInput = document.getElementById('target_value');
    
    if (targetType === 'all') {
        targetValueGroup.style.display = 'none';
        targetValueInput.required = false;
    } else {
        targetValueGroup.style.display = 'block';
        targetValueInput.required = true;
    }
}

// Önizleme güncelleme
function updatePreview() {
    const title = document.getElementById('title').value || 'Bildirim Başlığı';
    const body = document.getElementById('body').value || 'Bildirim mesajı burada görünecek';
    
    document.getElementById('preview-title').textContent = title;
    document.getElementById('preview-body').textContent = body;
}

// Karakter sayacı
function updateCharCount() {
    const body = document.getElementById('body');
    const charCount = document.querySelector('.char-count');
    charCount.textContent = `${body.value.length}/500 karakter`;
}

// Form temizleme
function resetForm() {
    document.querySelector('.notification-form').reset();
    document.getElementById('template_select').value = '';
    document.getElementById('template_id').value = '';
    toggleTargetValue();
    updatePreview();
    updateCharCount();
}

// Event listeners
document.getElementById('title').addEventListener('input', updatePreview);
document.getElementById('body').addEventListener('input', function() {
    updatePreview();
    updateCharCount();
});

// Firebase ayarları modalını aç/kapat
function toggleFirebaseSettings() {
    const modal = document.getElementById('firebaseSettingsModal');
    if (modal.style.display === 'none' || modal.style.display === '') {
        modal.style.display = 'flex';
        loadCurrentFirebaseSettings();
    } else {
        modal.style.display = 'none';
    }
}

// Mevcut Firebase ayarlarını yükle
function loadCurrentFirebaseSettings() {
    fetch('firebase-settings.php?action=get', {
        method: 'GET',
        headers: {
            'Content-Type': 'application/json',
            'X-Requested-With': 'XMLHttpRequest'
        },
        credentials: 'same-origin'
    })
    .then(response => {
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return response.text();
    })
    .then(text => {
        console.log('Raw response:', text);
        try {
            const data = JSON.parse(text);
            if (data.success && data.config) {
                document.getElementById('project_id').value = data.config.project_id || '';
                // Service account JSON'ı güvenlik nedeniyle göstermiyoruz
            }
        } catch (e) {
            console.error('JSON parse error:', e);
            console.error('Response text:', text);
            throw new Error('Invalid JSON response');
        }
    })
    .catch(error => {
        console.error('Firebase ayarları yüklenemedi:', error);
    });
}

// Sayfa yüklendiğinde
document.addEventListener('DOMContentLoaded', function() {
    updatePreview();
    updateCharCount();
    toggleTargetValue();
});
</script>

<style>
.notification-container {
    display: grid;
    grid-template-columns: 2fr 1fr;
    gap: 2rem;
    margin-top: 1rem;
}

.form-card, .preview-card, .recent-notifications {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
}

.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1rem;
}

.form-group {
    margin-bottom: 1rem;
}

.form-group label {
    display: block;
    margin-bottom: 0.5rem;
    font-weight: 600;
    color: #333;
}

.form-group input, .form-group select, .form-group textarea {
    width: 100%;
    padding: 0.75rem;
    border: 2px solid #e1e5e9;
    border-radius: 8px;
    font-size: 0.9rem;
    transition: border-color 0.3s;
}

.form-group input:focus, .form-group select:focus, .form-group textarea:focus {
    outline: none;
    border-color: #4f46e5;
}

.radio-group {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.radio-option {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
}

.char-count {
    color: #666;
    font-size: 0.8rem;
    margin-top: 0.25rem;
}

.form-actions {
    display: flex;
    gap: 1rem;
    margin-top: 2rem;
}

.btn {
    padding: 0.75rem 1.5rem;
    border: none;
    border-radius: 8px;
    font-weight: 600;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    transition: all 0.3s;
}

.btn-primary {
    background: linear-gradient(135deg, #4f46e5, #7c3aed);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(79, 70, 229, 0.4);
}

.btn-secondary {
    background: #f3f4f6;
    color: #374151;
}

.btn-secondary:hover {
    background: #e5e7eb;
}

.phone-mockup {
    background: #1f2937;
    border-radius: 20px;
    padding: 1rem;
    margin: 1rem 0;
}

.notification-preview {
    background: white;
    border-radius: 12px;
    padding: 1rem;
    display: flex;
    align-items: flex-start;
    gap: 0.75rem;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}

.notification-icon {
    width: 40px;
    height: 40px;
    background: #4f46e5;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    flex-shrink: 0;
}

.notification-content {
    flex: 1;
}

.notification-title {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.25rem;
}

.notification-body {
    color: #6b7280;
    font-size: 0.9rem;
    line-height: 1.4;
    margin-bottom: 0.5rem;
}

.notification-time {
    color: #9ca3af;
    font-size: 0.8rem;
}

.recent-list {
    max-height: 300px;
    overflow-y: auto;
}

.recent-item {
    padding: 1rem;
    border-left: 4px solid #e5e7eb;
    margin-bottom: 0.75rem;
    background: #f9fafb;
    border-radius: 0 8px 8px 0;
}

.recent-item.sent {
    border-left-color: #10b981;
}

.recent-item.failed {
    border-left-color: #ef4444;
}

.recent-title {
    font-weight: 600;
    color: #1f2937;
    margin-bottom: 0.25rem;
}

.recent-body {
    color: #6b7280;
    font-size: 0.9rem;
    margin-bottom: 0.5rem;
}

.recent-meta {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    font-size: 0.8rem;
    color: #9ca3af;
}

.status-badge {
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
    font-weight: 600;
    text-transform: uppercase;
    font-size: 0.7rem;
}

.status-sent {
    background: #d1fae5;
    color: #065f46;
}

.status-failed {
    background: #fee2e2;
    color: #991b1b;
}

.status-pending {
    background: #fef3c7;
    color: #92400e;
}

/* Header Actions */
.page-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 2rem;
}

.header-actions {
    display: flex;
    gap: 1rem;
}

/* Modal Styles */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
}

.modal-content {
    background: white;
    border-radius: 12px;
    width: 90%;
    max-width: 600px;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
}

.modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1.5rem;
    border-bottom: 1px solid #e5e7eb;
}

.modal-header h3 {
    margin: 0;
    color: #1f2937;
    font-size: 1.25rem;
}

.modal-close {
    background: none;
    border: none;
    font-size: 1.5rem;
    color: #6b7280;
    cursor: pointer;
    padding: 0.5rem;
    border-radius: 50%;
    transition: all 0.3s;
}

.modal-close:hover {
    background: #f3f4f6;
    color: #374151;
}

.modal-body {
    padding: 1.5rem;
}

.modal-footer {
    display: flex;
    justify-content: flex-end;
    gap: 1rem;
    padding: 1.5rem;
    border-top: 1px solid #e5e7eb;
    background: #f9fafb;
    border-radius: 0 0 12px 12px;
}

/* Firebase Help Section */
.firebase-help {
    background: #f0f9ff;
    border: 1px solid #0ea5e9;
    border-radius: 8px;
    padding: 1rem;
    margin-top: 1rem;
}

.firebase-help h4 {
    color: #0c4a6e;
    margin: 0 0 0.75rem 0;
    font-size: 1rem;
}

.firebase-help ol {
    margin: 0;
    padding-left: 1.5rem;
    color: #0c4a6e;
}

.firebase-help li {
    margin-bottom: 0.5rem;
    line-height: 1.4;
}

.firebase-help a {
    color: #0ea5e9;
    text-decoration: none;
}

.firebase-help a:hover {
    text-decoration: underline;
}

/* Config Warning Styles */
.config-warning {
    background: linear-gradient(135deg, #fef3c7, #fde68a);
    border: 2px solid #f59e0b;
    border-radius: 12px;
    margin-bottom: 2rem;
    overflow: hidden;
}

.warning-content {
    display: flex;
    align-items: center;
    padding: 1.5rem;
    gap: 1.5rem;
}

.warning-icon {
    flex-shrink: 0;
    width: 60px;
    height: 60px;
    background: #f59e0b;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
}

.warning-text {
    flex: 1;
}

.warning-text h3 {
    margin: 0 0 0.5rem 0;
    color: #92400e;
    font-size: 1.25rem;
}

.warning-text p {
    margin: 0;
    color: #92400e;
    line-height: 1.5;
}

.warning-action {
    flex-shrink: 0;
}

.form-disabled-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(255, 255, 255, 0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 12px;
    z-index: 10;
}

.form-disabled-overlay p {
    background: #fee2e2;
    color: #991b1b;
    padding: 1rem 1.5rem;
    border-radius: 8px;
    margin: 0;
    font-weight: 600;
    text-align: center;
}

.form-card {
    position: relative;
}

@media (max-width: 768px) {
    .notification-container {
        grid-template-columns: 1fr;
    }
    
    .form-row {
        grid-template-columns: 1fr;
    }
    
    .radio-group {
        flex-direction: column;
    }
    
    .page-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .modal-content {
        width: 95%;
        margin: 1rem;
    }
    
    .modal-footer {
        flex-direction: column;
    }
    
    .warning-content {
        flex-direction: column;
        text-align: center;
        gap: 1rem;
    }
    
    .warning-icon {
        align-self: center;
    }
}
</style>

<?php include 'includes/footer.php'; ?>
