<?php
/**
 * Raporlar Sayfası
 * Admin Panel - Detaylı İstatistikler ve Raporlar
 */

require_once __DIR__ . '/config/database.php';

$page_title = "Raporlar";

// Header include et
require_once __DIR__ . '/includes/header.php';

// Veritabanı bağlantısı
$database = new Database();
$pdo = $database->getConnection();

// Genel İstatistikler
try {
    // Kullanıcı İstatistikleri
    $user_stats = $pdo->query("
        SELECT 
            COUNT(*) as total_users,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as new_users_30d,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as new_users_7d,
            SUM(CASE WHEN created_at >= CURDATE() THEN 1 ELSE 0 END) as new_users_today
        FROM dua_users
    ")->fetch(PDO::FETCH_ASSOC);

    // Dua Talepleri İstatistikleri
    $request_stats = $pdo->query("
        SELECT 
            COUNT(*) as total_requests,
            SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_requests,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_requests,
            SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected_requests,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as new_requests_30d,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as new_requests_7d,
            SUM(CASE WHEN created_at >= CURDATE() THEN 1 ELSE 0 END) as new_requests_today
        FROM dua_requests
    ")->fetch(PDO::FETCH_ASSOC);

    // Yorum İstatistikleri
    $comment_stats = $pdo->query("
        SELECT 
            COUNT(*) as total_comments,
            SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved_comments,
            SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending_comments,
            SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected_comments,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) THEN 1 ELSE 0 END) as new_comments_30d,
            SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) THEN 1 ELSE 0 END) as new_comments_7d,
            SUM(CASE WHEN created_at >= CURDATE() THEN 1 ELSE 0 END) as new_comments_today
        FROM dua_comments
    ")->fetch(PDO::FETCH_ASSOC);

    // Günlük Aktivite (Son 30 gün)
    $daily_activity = $pdo->query("
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as total_activity,
            'request' as type
        FROM dua_requests 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        
        UNION ALL
        
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as total_activity,
            'comment' as type
        FROM dua_comments 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        
        UNION ALL
        
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as total_activity,
            'user' as type
        FROM dua_users 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        
        ORDER BY date DESC
    ")->fetchAll(PDO::FETCH_ASSOC);

    // En Aktif Kullanıcılar
    $active_users = $pdo->query("
        SELECT 
            u.full_name,
            u.email,
            u.created_at as join_date,
            COUNT(DISTINCT r.id) as request_count,
            COUNT(DISTINCT c.id) as comment_count,
            (COUNT(DISTINCT r.id) + COUNT(DISTINCT c.id)) as total_activity
        FROM dua_users u
        LEFT JOIN dua_requests r ON u.id = r.user_id
        LEFT JOIN dua_comments c ON u.id = c.user_id
        GROUP BY u.id, u.full_name, u.email, u.created_at
        HAVING total_activity > 0
        ORDER BY total_activity DESC
        LIMIT 10
    ")->fetchAll(PDO::FETCH_ASSOC);

    // Popüler Dua Talepleri (En çok yorumlanan)
    $popular_requests = $pdo->query("
        SELECT 
            r.description,
            r.created_at,
            r.status,
            u.full_name as requester_name,
            COUNT(c.id) as comment_count
        FROM dua_requests r
        LEFT JOIN dua_users u ON r.user_id = u.id
        LEFT JOIN dua_comments c ON r.id = c.request_id
        WHERE r.status = 'approved'
        GROUP BY r.id, r.description, r.created_at, r.status, u.full_name
        HAVING comment_count > 0
        ORDER BY comment_count DESC
        LIMIT 10
    ")->fetchAll(PDO::FETCH_ASSOC);

} catch (PDOException $e) {
    $error_message = "Raporlar yüklenirken hata oluştu: " . $e->getMessage();
}
?>

<div class="page-header">
    <h2>Raporlar ve İstatistikler</h2>
    <p>Dua Kardeşliği platformunun detaylı analiz raporları</p>
</div>

<?php if (isset($error_message)): ?>
    <div class="alert alert-error"><?php echo $error_message; ?></div>
<?php else: ?>

<!-- Genel İstatistik Kartları -->
<div class="stats-overview">
    <div class="stats-grid">
        <!-- Kullanıcı İstatistikleri -->
        <div class="stat-card users">
            <div class="stat-icon">
                <i class="fas fa-users"></i>
            </div>
            <div class="stat-content">
                <h3><?php echo number_format($user_stats['total_users']); ?></h3>
                <p>Toplam Kullanıcı</p>
                <div class="stat-details">
                    <span class="stat-change positive">+<?php echo $user_stats['new_users_30d']; ?> (30 gün)</span>
                </div>
            </div>
        </div>

        <!-- Dua Talepleri İstatistikleri -->
        <div class="stat-card requests">
            <div class="stat-icon">
                <i class="fas fa-hands-praying"></i>
            </div>
            <div class="stat-content">
                <h3><?php echo number_format($request_stats['total_requests']); ?></h3>
                <p>Toplam Dua Talebi</p>
                <div class="stat-details">
                    <span class="stat-change positive">+<?php echo $request_stats['new_requests_30d']; ?> (30 gün)</span>
                </div>
            </div>
        </div>

        <!-- Yorum İstatistikleri -->
        <div class="stat-card comments">
            <div class="stat-icon">
                <i class="fas fa-comments"></i>
            </div>
            <div class="stat-content">
                <h3><?php echo number_format($comment_stats['total_comments']); ?></h3>
                <p>Toplam Yorum</p>
                <div class="stat-details">
                    <span class="stat-change positive">+<?php echo $comment_stats['new_comments_30d']; ?> (30 gün)</span>
                </div>
            </div>
        </div>

        <!-- Onay Oranı -->
        <div class="stat-card approval">
            <div class="stat-icon">
                <i class="fas fa-check-circle"></i>
            </div>
            <div class="stat-content">
                <h3><?php echo $request_stats['total_requests'] > 0 ? round(($request_stats['approved_requests'] / $request_stats['total_requests']) * 100, 1) : 0; ?>%</h3>
                <p>Dua Onay Oranı</p>
                <div class="stat-details">
                    <span class="stat-info"><?php echo $request_stats['approved_requests']; ?> / <?php echo $request_stats['total_requests']; ?> onaylandı</span>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Detaylı Raporlar -->
<div class="reports-grid">
    <!-- Durum Dağılımları -->
    <div class="report-section">
        <div class="report-card">
            <div class="report-header">
                <h3>Dua Talepleri Durum Dağılımı</h3>
                <i class="fas fa-chart-pie"></i>
            </div>
            <div class="report-content">
                <div class="status-breakdown">
                    <div class="status-item approved">
                        <div class="status-bar">
                            <div class="status-fill" style="width: <?php echo $request_stats['total_requests'] > 0 ? ($request_stats['approved_requests'] / $request_stats['total_requests']) * 100 : 0; ?>%"></div>
                        </div>
                        <div class="status-info">
                            <span class="status-label">Onaylanan</span>
                            <span class="status-count"><?php echo number_format($request_stats['approved_requests']); ?></span>
                        </div>
                    </div>
                    
                    <div class="status-item pending">
                        <div class="status-bar">
                            <div class="status-fill" style="width: <?php echo $request_stats['total_requests'] > 0 ? ($request_stats['pending_requests'] / $request_stats['total_requests']) * 100 : 0; ?>%"></div>
                        </div>
                        <div class="status-info">
                            <span class="status-label">Bekleyen</span>
                            <span class="status-count"><?php echo number_format($request_stats['pending_requests']); ?></span>
                        </div>
                    </div>
                    
                    <div class="status-item rejected">
                        <div class="status-bar">
                            <div class="status-fill" style="width: <?php echo $request_stats['total_requests'] > 0 ? ($request_stats['rejected_requests'] / $request_stats['total_requests']) * 100 : 0; ?>%"></div>
                        </div>
                        <div class="status-info">
                            <span class="status-label">Reddedilen</span>
                            <span class="status-count"><?php echo number_format($request_stats['rejected_requests']); ?></span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Yorum Durum Dağılımı -->
    <div class="report-section">
        <div class="report-card">
            <div class="report-header">
                <h3>Yorumlar Durum Dağılımı</h3>
                <i class="fas fa-chart-donut"></i>
            </div>
            <div class="report-content">
                <div class="status-breakdown">
                    <div class="status-item approved">
                        <div class="status-bar">
                            <div class="status-fill" style="width: <?php echo $comment_stats['total_comments'] > 0 ? ($comment_stats['approved_comments'] / $comment_stats['total_comments']) * 100 : 0; ?>%"></div>
                        </div>
                        <div class="status-info">
                            <span class="status-label">Onaylanan</span>
                            <span class="status-count"><?php echo number_format($comment_stats['approved_comments']); ?></span>
                        </div>
                    </div>
                    
                    <div class="status-item pending">
                        <div class="status-bar">
                            <div class="status-fill" style="width: <?php echo $comment_stats['total_comments'] > 0 ? ($comment_stats['pending_comments'] / $comment_stats['total_comments']) * 100 : 0; ?>%"></div>
                        </div>
                        <div class="status-info">
                            <span class="status-label">Bekleyen</span>
                            <span class="status-count"><?php echo number_format($comment_stats['pending_comments']); ?></span>
                        </div>
                    </div>
                    
                    <div class="status-item rejected">
                        <div class="status-bar">
                            <div class="status-fill" style="width: <?php echo $comment_stats['total_comments'] > 0 ? ($comment_stats['rejected_comments'] / $comment_stats['total_comments']) * 100 : 0; ?>%"></div>
                        </div>
                        <div class="status-info">
                            <span class="status-label">Reddedilen</span>
                            <span class="status-count"><?php echo number_format($comment_stats['rejected_comments']); ?></span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Haftalık Aktivite Özeti -->
<div class="activity-summary">
    <div class="report-card">
        <div class="report-header">
            <h3>Son 7 Gün Aktivite Özeti</h3>
            <i class="fas fa-calendar-week"></i>
        </div>
        <div class="report-content">
            <div class="activity-grid">
                <div class="activity-item">
                    <div class="activity-icon users">
                        <i class="fas fa-user-plus"></i>
                    </div>
                    <div class="activity-info">
                        <h4><?php echo number_format($user_stats['new_users_7d']); ?></h4>
                        <p>Yeni Kullanıcı</p>
                    </div>
                </div>
                
                <div class="activity-item">
                    <div class="activity-icon requests">
                        <i class="fas fa-plus-circle"></i>
                    </div>
                    <div class="activity-info">
                        <h4><?php echo number_format($request_stats['new_requests_7d']); ?></h4>
                        <p>Yeni Dua Talebi</p>
                    </div>
                </div>
                
                <div class="activity-item">
                    <div class="activity-icon comments">
                        <i class="fas fa-comment-plus"></i>
                    </div>
                    <div class="activity-info">
                        <h4><?php echo number_format($comment_stats['new_comments_7d']); ?></h4>
                        <p>Yeni Yorum</p>
                    </div>
                </div>
                
                <div class="activity-item">
                    <div class="activity-icon today">
                        <i class="fas fa-calendar-day"></i>
                    </div>
                    <div class="activity-info">
                        <h4><?php echo number_format($user_stats['new_users_today'] + $request_stats['new_requests_today'] + $comment_stats['new_comments_today']); ?></h4>
                        <p>Bugünkü Toplam Aktivite</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- En Aktif Kullanıcılar -->
<?php if (!empty($active_users)): ?>
<div class="report-section">
    <div class="report-card">
        <div class="report-header">
            <h3>En Aktif Kullanıcılar</h3>
            <i class="fas fa-trophy"></i>
        </div>
        <div class="report-content">
            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Sıra</th>
                            <th>Kullanıcı</th>
                            <th>Dua Talepleri</th>
                            <th>Yorumlar</th>
                            <th>Toplam Aktivite</th>
                            <th>Katılım Tarihi</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($active_users as $index => $user): ?>
                            <tr>
                                <td>
                                    <span class="rank-badge rank-<?php echo $index + 1; ?>">
                                        <?php echo $index + 1; ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="user-info">
                                        <strong><?php echo htmlspecialchars($user['full_name']); ?></strong>
                                        <br><small><?php echo htmlspecialchars($user['email']); ?></small>
                                    </div>
                                </td>
                                <td>
                                    <span class="activity-count requests"><?php echo $user['request_count']; ?></span>
                                </td>
                                <td>
                                    <span class="activity-count comments"><?php echo $user['comment_count']; ?></span>
                                </td>
                                <td>
                                    <span class="activity-count total"><?php echo $user['total_activity']; ?></span>
                                </td>
                                <td>
                                    <span class="date-info"><?php echo date('d.m.Y', strtotime($user['join_date'])); ?></span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Popüler Dua Talepleri -->
<?php if (!empty($popular_requests)): ?>
<div class="report-section">
    <div class="report-card">
        <div class="report-header">
            <h3>En Çok Yorumlanan Dua Talepleri</h3>
            <i class="fas fa-fire"></i>
        </div>
        <div class="report-content">
            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>Dua Talebi</th>
                            <th>Talep Eden</th>
                            <th>Yorum Sayısı</th>
                            <th>Tarih</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($popular_requests as $request): ?>
                            <tr>
                                <td>
                                    <div class="request-preview">
                                        <?php echo htmlspecialchars(mb_substr($request['description'], 0, 100)); ?>
                                        <?php if (mb_strlen($request['description']) > 100): ?>...<?php endif; ?>
                                    </div>
                                </td>
                                <td>
                                    <span class="user-name"><?php echo htmlspecialchars($request['requester_name'] ?: 'Anonim'); ?></span>
                                </td>
                                <td>
                                    <span class="comment-count-badge"><?php echo $request['comment_count']; ?></span>
                                </td>
                                <td>
                                    <span class="date-info"><?php echo date('d.m.Y', strtotime($request['created_at'])); ?></span>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<?php endif; ?>

<style>
.stats-overview {
    margin-bottom: 30px;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.stat-card {
    background: white;
    padding: 25px;
    border-radius: 12px;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    display: flex;
    align-items: center;
    gap: 20px;
    transition: transform 0.3s, box-shadow 0.3s;
}

.stat-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 15px rgba(0,0,0,0.15);
}

.stat-card.users { border-left: 5px solid #3498db; }
.stat-card.requests { border-left: 5px solid #e74c3c; }
.stat-card.comments { border-left: 5px solid #f39c12; }
.stat-card.approval { border-left: 5px solid #27ae60; }

.stat-icon {
    font-size: 2.5rem;
    padding: 15px;
    border-radius: 50%;
    color: white;
}

.stat-card.users .stat-icon { background: linear-gradient(135deg, #3498db, #2980b9); }
.stat-card.requests .stat-icon { background: linear-gradient(135deg, #e74c3c, #c0392b); }
.stat-card.comments .stat-icon { background: linear-gradient(135deg, #f39c12, #e67e22); }
.stat-card.approval .stat-icon { background: linear-gradient(135deg, #27ae60, #229954); }

.stat-content h3 {
    margin: 0 0 5px 0;
    font-size: 2.2rem;
    font-weight: bold;
    color: #2c3e50;
}

.stat-content p {
    margin: 0 0 10px 0;
    color: #7f8c8d;
    font-size: 1rem;
}

.stat-details {
    font-size: 0.9rem;
}

.stat-change {
    padding: 3px 8px;
    border-radius: 12px;
    font-weight: 500;
}

.stat-change.positive {
    background: #d4edda;
    color: #155724;
}

.stat-info {
    color: #6c757d;
}

.reports-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.report-section {
    margin-bottom: 30px;
}

.report-card {
    background: white;
    border-radius: 12px;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    overflow: hidden;
}

.report-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.report-header h3 {
    margin: 0;
    font-size: 1.2rem;
}

.report-header i {
    font-size: 1.5rem;
    opacity: 0.8;
}

.report-content {
    padding: 25px;
}

.status-breakdown {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.status-item {
    display: flex;
    align-items: center;
    gap: 15px;
}

.status-bar {
    flex: 1;
    height: 8px;
    background: #ecf0f1;
    border-radius: 4px;
    overflow: hidden;
}

.status-fill {
    height: 100%;
    border-radius: 4px;
    transition: width 0.8s ease;
}

.status-item.approved .status-fill { background: linear-gradient(90deg, #27ae60, #2ecc71); }
.status-item.pending .status-fill { background: linear-gradient(90deg, #f39c12, #f1c40f); }
.status-item.rejected .status-fill { background: linear-gradient(90deg, #e74c3c, #ec7063); }

.status-info {
    display: flex;
    flex-direction: column;
    align-items: flex-end;
    min-width: 100px;
}

.status-label {
    font-size: 0.9rem;
    color: #7f8c8d;
}

.status-count {
    font-size: 1.1rem;
    font-weight: bold;
    color: #2c3e50;
}

.activity-summary {
    margin-bottom: 30px;
}

.activity-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
}

.activity-item {
    display: flex;
    align-items: center;
    gap: 15px;
    padding: 20px;
    background: #f8f9fa;
    border-radius: 8px;
    border-left: 4px solid #dee2e6;
}

.activity-icon {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.2rem;
}

.activity-icon.users { background: linear-gradient(135deg, #3498db, #2980b9); }
.activity-icon.requests { background: linear-gradient(135deg, #e74c3c, #c0392b); }
.activity-icon.comments { background: linear-gradient(135deg, #f39c12, #e67e22); }
.activity-icon.today { background: linear-gradient(135deg, #9b59b6, #8e44ad); }

.activity-info h4 {
    margin: 0;
    font-size: 1.8rem;
    font-weight: bold;
    color: #2c3e50;
}

.activity-info p {
    margin: 0;
    color: #7f8c8d;
    font-size: 0.9rem;
}

.table-container {
    overflow-x: auto;
}

.data-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 10px;
}

.data-table th,
.data-table td {
    padding: 12px;
    text-align: left;
    border-bottom: 1px solid #dee2e6;
}

.data-table th {
    background: #f8f9fa;
    font-weight: 600;
    color: #495057;
}

.rank-badge {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 30px;
    height: 30px;
    border-radius: 50%;
    color: white;
    font-weight: bold;
    font-size: 0.9rem;
}

.rank-badge.rank-1 { background: linear-gradient(135deg, #f1c40f, #f39c12); }
.rank-badge.rank-2 { background: linear-gradient(135deg, #95a5a6, #7f8c8d); }
.rank-badge.rank-3 { background: linear-gradient(135deg, #e67e22, #d35400); }
.rank-badge:not(.rank-1):not(.rank-2):not(.rank-3) { background: linear-gradient(135deg, #bdc3c7, #95a5a6); }

.user-info strong {
    color: #2c3e50;
}

.user-info small {
    color: #7f8c8d;
}

.activity-count {
    display: inline-block;
    padding: 4px 8px;
    border-radius: 12px;
    font-weight: 500;
    font-size: 0.9rem;
}

.activity-count.requests {
    background: #ffeaa7;
    color: #d63031;
}

.activity-count.comments {
    background: #fab1a0;
    color: #e17055;
}

.activity-count.total {
    background: #a29bfe;
    color: #6c5ce7;
}

.date-info {
    color: #7f8c8d;
    font-size: 0.9rem;
}

.request-preview {
    max-width: 300px;
    line-height: 1.4;
    color: #2c3e50;
}

.user-name {
    color: #2c3e50;
    font-weight: 500;
}

.comment-count-badge {
    display: inline-block;
    background: linear-gradient(135deg, #74b9ff, #0984e3);
    color: white;
    padding: 6px 12px;
    border-radius: 20px;
    font-weight: bold;
    font-size: 0.9rem;
}

@media (max-width: 768px) {
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .reports-grid {
        grid-template-columns: 1fr;
    }
    
    .activity-grid {
        grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    }
    
    .stat-card {
        padding: 20px;
    }
    
    .stat-icon {
        font-size: 2rem;
        padding: 12px;
    }
    
    .stat-content h3 {
        font-size: 1.8rem;
    }
}
</style>

<?php require_once 'includes/footer.php'; ?>
