<?php
/**
 * Detaylı İstatistikler Sayfası
 * Admin Panel - Kapsamlı analiz ve raporlar
 */

session_start();
require_once __DIR__ . '/config/database.php';

$page_title = 'Detaylı İstatistikler';

// Admin session kontrolü
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit();
}

// Database bağlantısı
$database = new Database();
$db = $database->getConnection();

// Tarih aralığı parametreleri
$date_range = $_GET['range'] ?? '30';
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');

try {
    // Genel İstatistikler
    $general_stats = [];
    
    // Toplam story sayıları
    $stmt = $db->query("
        SELECT 
            COUNT(*) as total_stories,
            SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active_stories,
            SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive_stories,
            SUM(view_count) as total_views,
            AVG(view_count) as avg_views_per_story
        FROM story_cards
    ");
    $general_stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Kategori bazlı istatistikler
    $category_stats = [];
    $stmt = $db->query("
        SELECT 
            sc.display_name,
            sc.name as category_key,
            COUNT(st.id) as story_count,
            SUM(st.view_count) as total_views,
            AVG(st.view_count) as avg_views,
            MAX(st.view_count) as max_views,
            SUM(CASE WHEN st.is_active = 1 THEN 1 ELSE 0 END) as active_count
        FROM story_categories sc
        LEFT JOIN story_cards st ON sc.id = st.category_id
        WHERE sc.is_active = 1
        GROUP BY sc.id, sc.display_name, sc.name
        ORDER BY story_count DESC
    ");
    $category_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // İçerik türü istatistikleri
    $content_type_stats = [];
    $stmt = $db->query("
        SELECT 
            content_type,
            COUNT(*) as count,
            SUM(view_count) as total_views,
            AVG(view_count) as avg_views
        FROM story_cards 
        WHERE is_active = 1
        GROUP BY content_type
    ");
    $content_type_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Günlük aktivite (son 30 gün)
    $daily_activity = [];
    $stmt = $db->prepare("
        SELECT 
            DATE(created_at) as date,
            COUNT(*) as story_count
        FROM story_cards 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        GROUP BY DATE(created_at)
        ORDER BY date DESC
    ");
    $stmt->execute();
    $daily_activity = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // En popüler story'ler
    $popular_stories = [];
    $stmt = $db->query("
        SELECT 
            st.id,
            st.title,
            st.content_type,
            st.view_count,
            st.created_at,
            sc.display_name as category_name
        FROM story_cards st
        JOIN story_categories sc ON st.category_id = sc.id
        WHERE st.is_active = 1
        ORDER BY st.view_count DESC
        LIMIT 10
    ");
    $popular_stories = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // En az görüntülenen story'ler
    $least_viewed_stories = [];
    $stmt = $db->query("
        SELECT 
            st.id,
            st.title,
            st.content_type,
            st.view_count,
            st.created_at,
            sc.display_name as category_name
        FROM story_cards st
        JOIN story_categories sc ON st.category_id = sc.id
        WHERE st.is_active = 1
        ORDER BY st.view_count ASC
        LIMIT 10
    ");
    $least_viewed_stories = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Aylık trend analizi
    $monthly_trends = [];
    $stmt = $db->query("
        SELECT 
            DATE_FORMAT(created_at, '%Y-%m') as month,
            COUNT(*) as story_count,
            SUM(view_count) as total_views
        FROM story_cards 
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
        GROUP BY DATE_FORMAT(created_at, '%Y-%m')
        ORDER BY month DESC
    ");
    $monthly_trends = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Admin aktivite istatistikleri
    $admin_stats = [];
    $stmt = $db->query("
        SELECT 
            au.full_name,
            au.username,
            COUNT(st.id) as story_count,
            SUM(st.view_count) as total_views,
            MAX(st.created_at) as last_story_date
        FROM admin_users au
        LEFT JOIN story_cards st ON au.id = st.created_by
        GROUP BY au.id, au.full_name, au.username
        ORDER BY story_count DESC
    ");
    $admin_stats = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Performans metrikleri
    $performance_metrics = [];
    $stmt = $db->query("
        SELECT 
            COUNT(*) as total_stories,
            SUM(view_count) as total_views,
            AVG(view_count) as avg_views_per_story,
            MAX(view_count) as max_views,
            MIN(view_count) as min_views,
            STD(view_count) as view_std_deviation
        FROM story_cards 
        WHERE is_active = 1
    ");
    $performance_metrics = $stmt->fetch(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    error_log("İstatistik getirme hatası: " . $e->getMessage());
    $_SESSION['error_message'] = 'İstatistikler yüklenirken hata oluştu: ' . $e->getMessage();
}

include 'includes/header.php';
?>

<div class="page-header">
    <h2>Detaylı İstatistikler</h2>
    <p>Story platformunuzun kapsamlı analiz ve performans raporları</p>
</div>

<!-- Filtre Seçenekleri -->
<div class="filters-section">
    <div class="filter-card">
        <div class="filter-header">
            <h3><i class="fas fa-filter"></i> Analiz Filtreleri</h3>
        </div>
        <div class="filter-body">
            <form method="GET" class="filter-form">
                <div class="filter-group">
                    <label for="range">Hızlı Seçim:</label>
                    <select name="range" id="range" onchange="this.form.submit()">
                        <option value="7" <?php echo $date_range === '7' ? 'selected' : ''; ?>>Son 7 Gün</option>
                        <option value="30" <?php echo $date_range === '30' ? 'selected' : ''; ?>>Son 30 Gün</option>
                        <option value="90" <?php echo $date_range === '90' ? 'selected' : ''; ?>>Son 3 Ay</option>
                        <option value="365" <?php echo $date_range === '365' ? 'selected' : ''; ?>>Son 1 Yıl</option>
                        <option value="custom" <?php echo $date_range === 'custom' ? 'selected' : ''; ?>>Özel Tarih</option>
                    </select>
                </div>
                
                <div class="filter-group custom-date" style="<?php echo $date_range !== 'custom' ? 'display: none;' : ''; ?>">
                    <label for="start_date">Başlangıç:</label>
                    <input type="date" name="start_date" id="start_date" value="<?php echo $start_date; ?>">
                </div>
                
                <div class="filter-group custom-date" style="<?php echo $date_range !== 'custom' ? 'display: none;' : ''; ?>">
                    <label for="end_date">Bitiş:</label>
                    <input type="date" name="end_date" id="end_date" value="<?php echo $end_date; ?>">
                </div>
                
                <div class="filter-actions">
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-chart-line"></i>
                        Analiz Et
                    </button>
                    <a href="statistics.php" class="btn btn-secondary">
                        <i class="fas fa-refresh"></i>
                        Sıfırla
                    </a>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Genel İstatistikler -->
<div class="stats-overview">
    <div class="stats-grid">
        <div class="stat-card primary">
            <div class="stat-icon">
                <i class="fas fa-images"></i>
            </div>
            <div class="stat-content">
                <h3><?php echo number_format($general_stats['total_stories'] ?? 0); ?></h3>
                <p>Toplam Story</p>
                <div class="stat-detail">
                    <span class="active"><?php echo number_format($general_stats['active_stories'] ?? 0); ?> aktif</span>
                    <span class="inactive"><?php echo number_format($general_stats['inactive_stories'] ?? 0); ?> pasif</span>
                </div>
            </div>
        </div>
        
        <div class="stat-card success">
            <div class="stat-icon">
                <i class="fas fa-eye"></i>
            </div>
            <div class="stat-content">
                <h3><?php echo number_format($general_stats['total_views'] ?? 0); ?></h3>
                <p>Toplam Görüntülenme</p>
                <div class="stat-detail">
                    <span>Ortalama: <?php echo number_format($general_stats['avg_views_per_story'] ?? 0, 1); ?></span>
                </div>
            </div>
        </div>
        
        <div class="stat-card info">
            <div class="stat-icon">
                <i class="fas fa-tags"></i>
            </div>
            <div class="stat-content">
                <h3><?php echo count($category_stats); ?></h3>
                <p>Aktif Kategori</p>
                <div class="stat-detail">
                    <span>Story dağılımı</span>
                </div>
            </div>
        </div>
        
        <div class="stat-card warning">
            <div class="stat-icon">
                <i class="fas fa-chart-line"></i>
            </div>
            <div class="stat-content">
                <h3><?php echo number_format($performance_metrics['view_std_deviation'] ?? 0, 1); ?></h3>
                <p>Görüntülenme Sapması</p>
                <div class="stat-detail">
                    <span>Performans metriği</span>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Kategori Analizi -->
<div class="analysis-section">
    <div class="analysis-grid">
        <div class="analysis-card">
            <div class="card-header">
                <h3><i class="fas fa-chart-pie"></i> Kategori Performans Analizi</h3>
            </div>
            <div class="card-body">
                <?php if (!empty($category_stats)): ?>
                    <div class="category-analysis">
                        <?php foreach ($category_stats as $category): ?>
                            <div class="category-item">
                                <div class="category-header">
                                    <h4><?php echo htmlspecialchars($category['display_name']); ?></h4>
                                    <span class="story-count"><?php echo $category['story_count']; ?> story</span>
                                </div>
                                
                                <div class="category-metrics">
                                    <div class="metric">
                                        <span class="metric-label">Toplam Görüntülenme:</span>
                                        <span class="metric-value"><?php echo number_format($category['total_views'] ?? 0); ?></span>
                                    </div>
                                    <div class="metric">
                                        <span class="metric-label">Ortalama Görüntülenme:</span>
                                        <span class="metric-value"><?php echo number_format($category['avg_views'] ?? 0, 1); ?></span>
                                    </div>
                                    <div class="metric">
                                        <span class="metric-label">En Yüksek Görüntülenme:</span>
                                        <span class="metric-value"><?php echo number_format($category['max_views'] ?? 0); ?></span>
                                    </div>
                                    <div class="metric">
                                        <span class="metric-label">Aktif Story Oranı:</span>
                                        <span class="metric-value">
                                            <?php 
                                            $active_ratio = $category['story_count'] > 0 ? 
                                                ($category['active_count'] / $category['story_count']) * 100 : 0;
                                            echo number_format($active_ratio, 1) . '%';
                                            ?>
                                        </span>
                                    </div>
                                </div>
                                
                                <div class="category-progress">
                                    <div class="progress-bar">
                                        <div class="progress-fill" style="width: <?php echo min(100, ($category['total_views'] / max(1, $general_stats['total_views'])) * 100); ?>%"></div>
                                    </div>
                                    <span class="progress-text">
                                        <?php echo number_format(($category['total_views'] / max(1, $general_stats['total_views'])) * 100, 1); ?>% toplam görüntülenme
                                    </span>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="no-data">
                        <i class="fas fa-chart-pie"></i>
                        <p>Kategori analizi için veri bulunamadı</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- İçerik Türü Analizi -->
        <div class="analysis-card">
            <div class="card-header">
                <h3><i class="fas fa-layer-group"></i> İçerik Türü Dağılımı</h3>
            </div>
            <div class="card-body">
                <?php if (!empty($content_type_stats)): ?>
                    <div class="content-type-analysis">
                        <?php foreach ($content_type_stats as $type): ?>
                            <div class="type-item">
                                <div class="type-header">
                                    <div class="type-icon">
                                        <?php if ($type['content_type'] === 'image'): ?>
                                            <i class="fas fa-image"></i>
                                        <?php else: ?>
                                            <i class="fas fa-font"></i>
                                        <?php endif; ?>
                                    </div>
                                    <div class="type-info">
                                        <h4><?php echo $type['content_type'] === 'image' ? 'Görsel' : 'Metin'; ?></h4>
                                        <span class="type-count"><?php echo number_format($type['count']); ?> story</span>
                                    </div>
                                </div>
                                
                                <div class="type-stats">
                                    <div class="stat-item">
                                        <span class="stat-label">Toplam Görüntülenme</span>
                                        <span class="stat-value"><?php echo number_format($type['total_views']); ?></span>
                                    </div>
                                    <div class="stat-item">
                                        <span class="stat-label">Ortalama Görüntülenme</span>
                                        <span class="stat-value"><?php echo number_format($type['avg_views'], 1); ?></span>
                                    </div>
                                </div>
                                
                                <div class="type-chart">
                                    <div class="chart-bar">
                                        <div class="chart-fill" style="width: <?php echo ($type['count'] / max(1, $general_stats['total_stories'])) * 100; ?>%"></div>
                                    </div>
                                    <span class="chart-percentage">
                                        <?php echo number_format(($type['count'] / max(1, $general_stats['total_stories'])) * 100, 1); ?>%
                                    </span>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="no-data">
                        <i class="fas fa-layer-group"></i>
                        <p>İçerik türü analizi için veri bulunamadı</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Popüler İçerik Analizi -->
<div class="content-analysis">
    <div class="content-grid">
        <!-- En Popüler Story'ler -->
        <div class="content-card">
            <div class="card-header">
                <h3><i class="fas fa-fire"></i> En Popüler Story'ler</h3>
            </div>
            <div class="card-body">
                <?php if (!empty($popular_stories)): ?>
                    <div class="story-list">
                        <?php foreach ($popular_stories as $index => $story): ?>
                            <div class="story-item">
                                <div class="story-rank">
                                    <span class="rank-number"><?php echo $index + 1; ?></span>
                                </div>
                                <div class="story-info">
                                    <h4 class="story-title">
                                        <?php echo htmlspecialchars(mb_substr($story['title'], 0, 50)); ?>
                                        <?php if (mb_strlen($story['title']) > 50): ?>...<?php endif; ?>
                                    </h4>
                                    <div class="story-meta">
                                        <span class="story-category"><?php echo htmlspecialchars($story['category_name']); ?></span>
                                        <span class="story-type">
                                            <?php if ($story['content_type'] === 'image'): ?>
                                                <i class="fas fa-image"></i> Görsel
                                            <?php else: ?>
                                                <i class="fas fa-font"></i> Metin
                                            <?php endif; ?>
                                        </span>
                                        <span class="story-date"><?php echo date('d.m.Y', strtotime($story['created_at'])); ?></span>
                                    </div>
                                </div>
                                <div class="story-views">
                                    <span class="view-count"><?php echo number_format($story['view_count']); ?></span>
                                    <span class="view-label">görüntülenme</span>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="no-data">
                        <i class="fas fa-fire"></i>
                        <p>Popüler story verisi bulunamadı</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        
        <!-- En Az Görüntülenen Story'ler -->
        <div class="content-card">
            <div class="card-header">
                <h3><i class="fas fa-eye-slash"></i> Geliştirilmesi Gereken Story'ler</h3>
            </div>
            <div class="card-body">
                <?php if (!empty($least_viewed_stories)): ?>
                    <div class="story-list">
                        <?php foreach ($least_viewed_stories as $index => $story): ?>
                            <div class="story-item low-performance">
                                <div class="story-rank">
                                    <span class="rank-number low"><?php echo $index + 1; ?></span>
                                </div>
                                <div class="story-info">
                                    <h4 class="story-title">
                                        <?php echo htmlspecialchars(mb_substr($story['title'], 0, 50)); ?>
                                        <?php if (mb_strlen($story['title']) > 50): ?>...<?php endif; ?>
                                    </h4>
                                    <div class="story-meta">
                                        <span class="story-category"><?php echo htmlspecialchars($story['category_name']); ?></span>
                                        <span class="story-type">
                                            <?php if ($story['content_type'] === 'image'): ?>
                                                <i class="fas fa-image"></i> Görsel
                                            <?php else: ?>
                                                <i class="fas fa-font"></i> Metin
                                            <?php endif; ?>
                                        </span>
                                        <span class="story-date"><?php echo date('d.m.Y', strtotime($story['created_at'])); ?></span>
                                    </div>
                                </div>
                                <div class="story-views low">
                                    <span class="view-count"><?php echo number_format($story['view_count']); ?></span>
                                    <span class="view-label">görüntülenme</span>
                                    <a href="story-edit.php?id=<?php echo $story['id']; ?>" class="improve-btn" title="Düzenle">
                                        <i class="fas fa-edit"></i>
                                    </a>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="no-data">
                        <i class="fas fa-eye-slash"></i>
                        <p>Düşük performanslı story verisi bulunamadı</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Aylık Trend Analizi -->
<?php if (!empty($monthly_trends)): ?>
<div class="trend-analysis">
    <div class="trend-card">
        <div class="card-header">
            <h3><i class="fas fa-chart-line"></i> Aylık Trend Analizi (Son 12 Ay)</h3>
        </div>
        <div class="card-body">
            <div class="trend-chart">
                <div class="chart-container">
                    <?php 
                    $max_stories = max(array_column($monthly_trends, 'story_count'));
                    $max_views = max(array_column($monthly_trends, 'total_views'));
                    ?>
                    <?php foreach (array_reverse($monthly_trends) as $trend): ?>
                        <div class="chart-item">
                            <div class="chart-bars">
                                <div class="bar story-bar" style="height: <?php echo ($trend['story_count'] / max(1, $max_stories)) * 100; ?>%" 
                                     title="<?php echo $trend['story_count']; ?> story"></div>
                                <div class="bar view-bar" style="height: <?php echo ($trend['total_views'] / max(1, $max_views)) * 100; ?>%" 
                                     title="<?php echo number_format($trend['total_views']); ?> görüntülenme"></div>
                            </div>
                            <div class="chart-label">
                                <?php echo date('M Y', strtotime($trend['month'] . '-01')); ?>
                            </div>
                            <div class="chart-values">
                                <span class="story-value"><?php echo $trend['story_count']; ?></span>
                                <span class="view-value"><?php echo number_format($trend['total_views']); ?></span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
                <div class="chart-legend">
                    <div class="legend-item">
                        <div class="legend-color story"></div>
                        <span>Story Sayısı</span>
                    </div>
                    <div class="legend-item">
                        <div class="legend-color view"></div>
                        <span>Görüntülenme</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<!-- Admin Aktivite Analizi -->
<?php if (!empty($admin_stats)): ?>
<div class="admin-analysis">
    <div class="admin-card">
        <div class="card-header">
            <h3><i class="fas fa-users-cog"></i> Admin Aktivite Analizi</h3>
        </div>
        <div class="card-body">
            <div class="admin-list">
                <?php foreach ($admin_stats as $admin): ?>
                    <div class="admin-item">
                        <div class="admin-avatar">
                            <?php echo strtoupper(substr($admin['full_name'], 0, 2)); ?>
                        </div>
                        <div class="admin-info">
                            <h4><?php echo htmlspecialchars($admin['full_name']); ?></h4>
                            <span class="admin-username">@<?php echo htmlspecialchars($admin['username']); ?></span>
                        </div>
                        <div class="admin-stats">
                            <div class="admin-stat">
                                <span class="stat-value"><?php echo number_format($admin['story_count']); ?></span>
                                <span class="stat-label">Story</span>
                            </div>
                            <div class="admin-stat">
                                <span class="stat-value"><?php echo number_format($admin['total_views']); ?></span>
                                <span class="stat-label">Görüntülenme</span>
                            </div>
                            <?php if ($admin['last_story_date']): ?>
                                <div class="admin-stat">
                                    <span class="stat-value"><?php echo date('d.m.Y', strtotime($admin['last_story_date'])); ?></span>
                                    <span class="stat-label">Son Story</span>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
</div>
<?php endif; ?>

<script>
// Tarih aralığı değiştiğinde özel tarih alanlarını göster/gizle
document.getElementById('range').addEventListener('change', function() {
    const customDateFields = document.querySelectorAll('.custom-date');
    if (this.value === 'custom') {
        customDateFields.forEach(field => field.style.display = 'block');
    } else {
        customDateFields.forEach(field => field.style.display = 'none');
    }
});

// Sayfa yüklendiğinde çalıştır
document.addEventListener('DOMContentLoaded', function() {
    const rangeSelect = document.getElementById('range');
    if (rangeSelect.value === 'custom') {
        document.querySelectorAll('.custom-date').forEach(field => field.style.display = 'block');
    }
});
</script>

<style>
.filters-section {
    margin-bottom: 2rem;
}

.filter-card {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    overflow: hidden;
}

.filter-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 1rem 1.5rem;
}

.filter-header h3 {
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.filter-body {
    padding: 1.5rem;
}

.filter-form {
    display: flex;
    gap: 1.5rem;
    align-items: end;
    flex-wrap: wrap;
}

.filter-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.filter-group label {
    font-weight: 600;
    color: #374151;
    font-size: 0.9rem;
}

.filter-group select,
.filter-group input {
    padding: 0.5rem;
    border: 2px solid #e5e7eb;
    border-radius: 6px;
    font-size: 0.9rem;
}

.filter-actions {
    display: flex;
    gap: 0.5rem;
}

.stats-overview {
    margin-bottom: 2rem;
}

.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
    gap: 1.5rem;
}

.stat-card {
    background: white;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    display: flex;
    align-items: center;
    gap: 1rem;
    position: relative;
    overflow: hidden;
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: var(--accent-color);
}

.stat-card.primary {
    --accent-color: #4f46e5;
}

.stat-card.success {
    --accent-color: #10b981;
}

.stat-card.info {
    --accent-color: #06b6d4;
}

.stat-card.warning {
    --accent-color: #f59e0b;
}

.stat-icon {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: var(--accent-color);
    color: white;
    font-size: 1.5rem;
}

.stat-content h3 {
    margin: 0 0 0.5rem 0;
    font-size: 2rem;
    font-weight: 700;
    color: #1f2937;
}

.stat-content p {
    margin: 0 0 0.5rem 0;
    color: #6b7280;
    font-weight: 600;
}

.stat-detail {
    display: flex;
    gap: 1rem;
    font-size: 0.8rem;
}

.stat-detail .active {
    color: #10b981;
    font-weight: 600;
}

.stat-detail .inactive {
    color: #ef4444;
    font-weight: 600;
}

.analysis-section {
    margin-bottom: 2rem;
}

.analysis-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
}

.analysis-card, .content-card, .trend-card, .admin-card {
    background: white;
    border-radius: 12px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    overflow: hidden;
}

.card-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 1.5rem;
}

.card-header h3 {
    margin: 0;
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 1.25rem;
}

.card-body {
    padding: 1.5rem;
}

.category-analysis {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.category-item {
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 1rem;
    background: #f9fafb;
}

.category-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.category-header h4 {
    margin: 0;
    color: #1f2937;
}

.story-count {
    background: #4f46e5;
    color: white;
    padding: 0.25rem 0.75rem;
    border-radius: 20px;
    font-size: 0.8rem;
    font-weight: 600;
}

.category-metrics {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 0.5rem;
    margin-bottom: 1rem;
}

.metric {
    display: flex;
    justify-content: space-between;
    font-size: 0.9rem;
}

.metric-label {
    color: #6b7280;
}

.metric-value {
    font-weight: 600;
    color: #1f2937;
}

.category-progress {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.progress-bar {
    flex: 1;
    height: 8px;
    background: #e5e7eb;
    border-radius: 4px;
    overflow: hidden;
}

.progress-fill {
    height: 100%;
    background: linear-gradient(90deg, #4f46e5, #7c3aed);
    transition: width 0.3s ease;
}

.progress-text {
    font-size: 0.8rem;
    color: #6b7280;
    white-space: nowrap;
}

.content-type-analysis {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.type-item {
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    padding: 1rem;
    background: #f9fafb;
}

.type-header {
    display: flex;
    align-items: center;
    gap: 1rem;
    margin-bottom: 1rem;
}

.type-icon {
    width: 40px;
    height: 40px;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    background: #4f46e5;
    color: white;
}

.type-info h4 {
    margin: 0 0 0.25rem 0;
    color: #1f2937;
}

.type-count {
    color: #6b7280;
    font-size: 0.9rem;
}

.type-stats {
    display: flex;
    gap: 2rem;
    margin-bottom: 1rem;
}

.stat-item {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
}

.stat-label {
    color: #6b7280;
    font-size: 0.8rem;
}

.stat-value {
    font-weight: 600;
    color: #1f2937;
}

.type-chart {
    display: flex;
    align-items: center;
    gap: 1rem;
}

.chart-bar {
    flex: 1;
    height: 6px;
    background: #e5e7eb;
    border-radius: 3px;
    overflow: hidden;
}

.chart-fill {
    height: 100%;
    background: linear-gradient(90deg, #10b981, #059669);
    transition: width 0.3s ease;
}

.chart-percentage {
    font-size: 0.9rem;
    font-weight: 600;
    color: #1f2937;
}

.content-analysis {
    margin-bottom: 2rem;
}

.content-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 2rem;
}

.story-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.story-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    background: #f9fafb;
}

.story-item.low-performance {
    border-color: #fbbf24;
    background: #fffbeb;
}

.story-rank {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    background: #4f46e5;
    color: white;
    font-weight: 700;
    flex-shrink: 0;
}

.rank-number.low {
    background: #f59e0b;
}

.story-info {
    flex: 1;
    min-width: 0;
}

.story-title {
    margin: 0 0 0.5rem 0;
    color: #1f2937;
    font-size: 1rem;
    line-height: 1.4;
}

.story-meta {
    display: flex;
    gap: 1rem;
    font-size: 0.8rem;
    color: #6b7280;
}

.story-category {
    background: #e5e7eb;
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
}

.story-views {
    text-align: right;
    flex-shrink: 0;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.25rem;
}

.story-views.low {
    position: relative;
}

.view-count {
    font-weight: 700;
    font-size: 1.1rem;
    color: #1f2937;
}

.view-label {
    font-size: 0.8rem;
    color: #6b7280;
}

.improve-btn {
    margin-top: 0.5rem;
    padding: 0.25rem 0.5rem;
    background: #f59e0b;
    color: white;
    border-radius: 4px;
    text-decoration: none;
    font-size: 0.8rem;
    transition: all 0.3s;
}

.improve-btn:hover {
    background: #d97706;
}

.trend-analysis {
    margin-bottom: 2rem;
}

.trend-chart {
    padding: 1rem;
}

.chart-container {
    display: flex;
    gap: 1rem;
    align-items: end;
    height: 200px;
    margin-bottom: 1rem;
}

.chart-item {
    flex: 1;
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 0.5rem;
}

.chart-bars {
    display: flex;
    gap: 2px;
    align-items: end;
    height: 150px;
}

.bar {
    width: 20px;
    border-radius: 2px 2px 0 0;
    transition: all 0.3s;
}

.story-bar {
    background: linear-gradient(to top, #4f46e5, #7c3aed);
}

.view-bar {
    background: linear-gradient(to top, #10b981, #059669);
}

.chart-label {
    font-size: 0.8rem;
    color: #6b7280;
    text-align: center;
}

.chart-values {
    display: flex;
    flex-direction: column;
    gap: 0.25rem;
    text-align: center;
}

.story-value {
    font-size: 0.8rem;
    color: #4f46e5;
    font-weight: 600;
}

.view-value {
    font-size: 0.8rem;
    color: #10b981;
    font-weight: 600;
}

.chart-legend {
    display: flex;
    justify-content: center;
    gap: 2rem;
}

.legend-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.legend-color {
    width: 16px;
    height: 16px;
    border-radius: 2px;
}

.legend-color.story {
    background: linear-gradient(135deg, #4f46e5, #7c3aed);
}

.legend-color.view {
    background: linear-gradient(135deg, #10b981, #059669);
}

.admin-analysis {
    margin-bottom: 2rem;
}

.admin-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.admin-item {
    display: flex;
    align-items: center;
    gap: 1rem;
    padding: 1rem;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    background: #f9fafb;
}

.admin-avatar {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: linear-gradient(135deg, #4f46e5, #7c3aed);
    color: white;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: 700;
    flex-shrink: 0;
}

.admin-info {
    flex: 1;
}

.admin-info h4 {
    margin: 0 0 0.25rem 0;
    color: #1f2937;
}

.admin-username {
    color: #6b7280;
    font-size: 0.9rem;
}

.admin-stats {
    display: flex;
    gap: 2rem;
}

.admin-stat {
    text-align: center;
}

.admin-stat .stat-value {
    display: block;
    font-weight: 700;
    color: #1f2937;
}

.admin-stat .stat-label {
    font-size: 0.8rem;
    color: #6b7280;
}

.no-data {
    text-align: center;
    padding: 3rem;
    color: #6b7280;
}

.no-data i {
    font-size: 3rem;
    margin-bottom: 1rem;
    color: #d1d5db;
}

.btn {
    padding: 0.5rem 1rem;
    border: none;
    border-radius: 6px;
    font-weight: 600;
    cursor: pointer;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    text-decoration: none;
    transition: all 0.3s;
}

.btn-primary {
    background: linear-gradient(135deg, #4f46e5, #7c3aed);
    color: white;
}

.btn-primary:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(79, 70, 229, 0.4);
}

.btn-secondary {
    background: #f3f4f6;
    color: #374151;
}

.btn-secondary:hover {
    background: #e5e7eb;
}

@media (max-width: 768px) {
    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }
    
    .stats-grid {
        grid-template-columns: 1fr;
    }
    
    .analysis-grid,
    .content-grid {
        grid-template-columns: 1fr;
    }
    
    .category-metrics {
        grid-template-columns: 1fr;
    }
    
    .type-stats {
        flex-direction: column;
        gap: 1rem;
    }
    
    .admin-stats {
        flex-direction: column;
        gap: 1rem;
    }
    
    .chart-container {
        height: 150px;
    }
    
    .chart-bars {
        height: 100px;
    }
    
    .chart-legend {
        flex-direction: column;
        gap: 1rem;
    }
}
</style>

<?php include 'includes/footer.php'; ?>
