<?php
/**
 * Yeni Story Ekleme Sayfası
 * Namaz Vakitleri Admin Panel
 */

$page_title = 'Yeni Story Ekle';
$header_actions = '<a href="story-cards.php" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Geri Dön</a>';

require_once 'config/database.php';
require_once 'includes/header.php';

// Form gönderildi mi?
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $title = trim($_POST['title'] ?? '');
    $content = trim($_POST['content'] ?? '');
    $category_id = (int)($_POST['category_id'] ?? 0);
    $content_type = $_POST['content_type'] ?? 'text';
    $background_color = $_POST['background_color'] ?? '#1a1a1a';
    $text_color = $_POST['text_color'] ?? '#ffffff';
    $font_size = (int)($_POST['font_size'] ?? 16);
    $is_active = isset($_POST['is_active']) ? 1 : 0;
    
    $errors = [];
    
    // Validasyon
    if (empty($title)) {
        $errors[] = 'Başlık gereklidir.';
    }
    
    if (empty($content) && $content_type === 'text') {
        $errors[] = 'Metin içeriği gereklidir.';
    }
    
    if ($category_id <= 0) {
        $errors[] = 'Kategori seçimi gereklidir.';
    }
    
    // Görsel yükleme kontrolü
    $image_path = null;
    if ($content_type === 'image' && isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $upload_dir = 'uploads/';
        if (!is_dir($upload_dir)) {
            mkdir($upload_dir, 0755, true);
        }
        
        $file_info = pathinfo($_FILES['image']['name']);
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        $file_extension = strtolower($file_info['extension']);
        
        if (!in_array($file_extension, $allowed_extensions)) {
            $errors[] = 'Sadece JPG, PNG, GIF ve WebP formatları desteklenir.';
        } else {
            // Dosya boyutu kontrolü (5MB)
            if ($_FILES['image']['size'] > 5 * 1024 * 1024) {
                $errors[] = 'Dosya boyutu 5MB\'dan küçük olmalıdır.';
            } else {
                // Görsel boyutu kontrolü
                $image_info = getimagesize($_FILES['image']['tmp_name']);
                if ($image_info === false) {
                    $errors[] = 'Geçersiz görsel dosyası.';
                } else {
                    $width = $image_info[0];
                    $height = $image_info[1];
                    
                    // 1080x1920 boyut kontrolü (tolerans %10)
                    if (abs($width - 1080) > 108 || abs($height - 1920) > 192) {
                        $errors[] = 'Görsel boyutu 1080x1920 piksel olmalıdır. (Mevcut: ' . $width . 'x' . $height . ')';
                    } else {
                        // Dosya adını oluştur
                        $filename = uniqid() . '_' . time() . '.' . $file_extension;
                        $target_path = $upload_dir . $filename;
                        
                        if (move_uploaded_file($_FILES['image']['tmp_name'], $target_path)) {
                            $image_path = $filename;
                        } else {
                            $errors[] = 'Dosya yüklenirken hata oluştu.';
                        }
                    }
                }
            }
        }
    } elseif ($content_type === 'image') {
        $errors[] = 'Görsel dosyası seçmelisiniz.';
    }
    
    // Hata yoksa kaydet
    if (empty($errors)) {
        try {
            $db = getDB();
            
            $sql = "INSERT INTO story_cards (
                category_id, title, content, content_type, image_path, 
                background_color, text_color, font_size, is_active, created_by
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                $category_id, $title, $content, $content_type, $image_path,
                $background_color, $text_color, $font_size, $is_active, $_SESSION['admin_id']
            ]);
            
            $_SESSION['success_message'] = 'Story başarıyla eklendi.';
            header('Location: story-cards.php');
            exit();
            
        } catch (PDOException $e) {
            $errors[] = 'Veritabanı hatası: ' . $e->getMessage();
        }
    }
}

// Kategorileri al
try {
    $db = getDB();
    $stmt = $db->query("SELECT * FROM story_categories WHERE is_active = 1 ORDER BY sort_order");
    $categories = $stmt->fetchAll();
} catch (PDOException $e) {
    $categories = [];
    $_SESSION['error_message'] = 'Kategoriler yüklenirken hata oluştu.';
}
?>

<!-- Hata mesajları -->
<?php if (!empty($errors)): ?>
    <div class="alert alert-error">
        <ul style="margin: 0; padding-left: 20px;">
            <?php foreach ($errors as $error): ?>
                <li><?php echo htmlspecialchars($error); ?></li>
            <?php endforeach; ?>
        </ul>
    </div>
<?php endif; ?>

<form method="POST" enctype="multipart/form-data" id="storyForm">
    <div class="row">
        <!-- Sol Kolon - Form Alanları -->
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h3><i class="fas fa-plus-circle"></i> Story Bilgileri</h3>
                </div>
                <div class="card-body">
                    <!-- Başlık -->
                    <div class="form-group">
                        <label for="title" class="form-label">
                            <i class="fas fa-heading"></i> Başlık *
                        </label>
                        <input type="text" id="title" name="title" class="form-control" 
                               value="<?php echo htmlspecialchars($_POST['title'] ?? ''); ?>" 
                               required maxlength="200" data-max-length="200">
                    </div>
                    
                    <!-- Kategori -->
                    <div class="form-group">
                        <label for="category_id" class="form-label">
                            <i class="fas fa-tags"></i> Kategori *
                        </label>
                        <select id="category_id" name="category_id" class="form-control" required>
                            <option value="">Kategori Seçin</option>
                            <?php foreach ($categories as $category): ?>
                                <option value="<?php echo $category['id']; ?>"
                                        <?php echo (isset($_POST['category_id']) && $_POST['category_id'] == $category['id']) ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($category['display_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- İçerik Türü -->
                    <div class="form-group">
                        <label class="form-label">
                            <i class="fas fa-layer-group"></i> İçerik Türü *
                        </label>
                        <div style="display: flex; gap: 20px; margin-top: 10px;">
                            <label style="display: flex; align-items: center; cursor: pointer;">
                                <input type="radio" name="content_type" value="text" 
                                       <?php echo (!isset($_POST['content_type']) || $_POST['content_type'] === 'text') ? 'checked' : ''; ?>
                                       style="margin-right: 8px;">
                                <i class="fas fa-font" style="margin-right: 5px;"></i> Metin
                            </label>
                            <label style="display: flex; align-items: center; cursor: pointer;">
                                <input type="radio" name="content_type" value="image" 
                                       <?php echo (isset($_POST['content_type']) && $_POST['content_type'] === 'image') ? 'checked' : ''; ?>
                                       style="margin-right: 8px;">
                                <i class="fas fa-image" style="margin-right: 5px;"></i> Görsel
                            </label>
                        </div>
                    </div>
                    
                    <!-- Metin İçeriği -->
                    <div class="form-group" id="textContent">
                        <label for="content" class="form-label">
                            <i class="fas fa-align-left"></i> İçerik *
                        </label>
                        <textarea id="content" name="content" class="form-control" rows="6" 
                                  maxlength="1000" data-max-length="1000"><?php echo htmlspecialchars($_POST['content'] ?? ''); ?></textarea>
                        <small class="text-muted">Maksimum 1000 karakter</small>
                    </div>
                    
                    <!-- Görsel Yükleme -->
                    <div class="form-group" id="imageContent" style="display: none;">
                        <label for="image" class="form-label">
                            <i class="fas fa-upload"></i> Görsel Dosyası * (1080x1920 piksel)
                        </label>
                        <div class="file-upload-container">
                            <input type="file" id="image" name="image" accept="image/*" 
                                   onchange="previewImage(this)" class="file-input">
                            <div class="file-upload-area" onclick="document.getElementById('image').click()">
                                <i class="fas fa-cloud-upload-alt" style="font-size: 32px; color: #007bff; margin-bottom: 15px;"></i>
                                <h4 style="margin: 10px 0; color: #333;">Görsel Dosyası Seçin</h4>
                                <p style="margin: 5px 0; color: #666;">Dosyayı buraya sürükleyip bırakın veya tıklayarak seçin</p>
                                <small style="color: #999;">JPG, PNG, GIF, WebP - Maksimum 5MB - 1080x1920 piksel</small>
                            </div>
                        </div>
                        <div id="imagePreviewContainer" style="display: none; margin-top: 15px;">
                            <img id="imagePreview" style="max-width: 200px; max-height: 300px; border-radius: 8px; border: 2px solid #ddd;">
                            <div style="margin-top: 10px;">
                                <button type="button" onclick="removeImage()" class="btn btn-danger btn-sm">
                                    <i class="fas fa-trash"></i> Görseli Kaldır
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Sağ Kolon - Tasarım ve Ayarlar -->
        <div class="col-md-4">
            <!-- Tasarım Ayarları -->
            <div class="card">
                <div class="card-header">
                    <h3><i class="fas fa-palette"></i> Tasarım Ayarları</h3>
                </div>
                <div class="card-body">
                    <!-- Arka Plan Rengi -->
                    <div class="form-group">
                        <label for="background_color" class="form-label">
                            <i class="fas fa-fill-drip"></i> Arka Plan Rengi
                        </label>
                        <input type="color" id="background_color" name="background_color" 
                               class="form-control" style="height: 50px;"
                               value="<?php echo $_POST['background_color'] ?? '#1a1a1a'; ?>">
                    </div>
                    
                    <!-- Metin Rengi -->
                    <div class="form-group">
                        <label for="text_color" class="form-label">
                            <i class="fas fa-font"></i> Metin Rengi
                        </label>
                        <input type="color" id="text_color" name="text_color" 
                               class="form-control" style="height: 50px;"
                               value="<?php echo $_POST['text_color'] ?? '#ffffff'; ?>">
                    </div>
                    
                    <!-- Font Boyutu -->
                    <div class="form-group">
                        <label for="font_size" class="form-label">
                            <i class="fas fa-text-height"></i> Font Boyutu
                        </label>
                        <select id="font_size" name="font_size" class="form-control">
                            <?php for ($size = 12; $size <= 24; $size += 2): ?>
                                <option value="<?php echo $size; ?>"
                                        <?php echo (isset($_POST['font_size']) && $_POST['font_size'] == $size) || (!isset($_POST['font_size']) && $size == 16) ? 'selected' : ''; ?>>
                                    <?php echo $size; ?>px
                                </option>
                            <?php endfor; ?>
                        </select>
                    </div>
                </div>
            </div>
            
            <!-- Durum ve İşlemler -->
            <div class="card">
                <div class="card-header">
                    <h3><i class="fas fa-cog"></i> Durum ve İşlemler</h3>
                </div>
                <div class="card-body">
                    <!-- Aktif/Pasif -->
                    <div class="form-group">
                        <label style="display: flex; align-items: center; cursor: pointer;">
                            <input type="checkbox" name="is_active" value="1" 
                                   <?php echo (!isset($_POST['is_active']) || $_POST['is_active']) ? 'checked' : ''; ?>
                                   style="margin-right: 10px;">
                            <i class="fas fa-eye" style="margin-right: 5px;"></i>
                            Story'yi aktif olarak yayınla
                        </label>
                    </div>
                    
                    <!-- Butonlar -->
                    <div style="display: grid; gap: 10px; margin-top: 30px;">
                        <button type="submit" class="btn btn-success">
                            <i class="fas fa-save"></i> Story'yi Kaydet
                        </button>
                        
                        <a href="story-cards.php" class="btn btn-secondary">
                            <i class="fas fa-times"></i> İptal Et
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- Önizleme -->
            <div class="card" id="previewCard">
                <div class="card-header">
                    <h3><i class="fas fa-eye"></i> Önizleme</h3>
                </div>
                <div class="card-body">
                    <div id="storyPreview" style="
                        width: 200px; 
                        height: 355px; 
                        border-radius: 10px; 
                        padding: 20px; 
                        display: flex; 
                        align-items: center; 
                        justify-content: center; 
                        text-align: center;
                        background: #1a1a1a;
                        color: #ffffff;
                        font-size: 16px;
                        line-height: 1.4;
                        overflow: hidden;
                        margin: 0 auto;
                    ">
                        <div>Story önizlemesi burada görünecek</div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</form>

<style>
.file-upload-container {
    margin-top: 10px;
}

.file-input {
    display: none;
}

.file-upload-area {
    border: 2px dashed #007bff;
    border-radius: 8px;
    padding: 40px 20px;
    text-align: center;
    cursor: pointer;
    transition: all 0.3s ease;
    background-color: #f8f9fa;
}

.file-upload-area:hover {
    border-color: #0056b3;
    background-color: #e3f2fd;
}

.file-upload-area.dragover {
    border-color: #28a745;
    background-color: #d4edda;
}

#imagePreviewContainer {
    text-align: center;
    padding: 15px;
    border: 1px solid #ddd;
    border-radius: 8px;
    background-color: #f8f9fa;
}
</style>

<script>
// jQuery olmadan vanilla JavaScript kullanıyoruz
document.addEventListener('DOMContentLoaded', function() {
    // İçerik türü değiştiğinde
    const contentTypeRadios = document.querySelectorAll('input[name="content_type"]');
    
    contentTypeRadios.forEach(function(radio) {
        radio.addEventListener('change', function() {
            var contentType = this.value;
            
            const textContent = document.getElementById('textContent');
            const imageContent = document.getElementById('imageContent');
            const contentTextarea = document.getElementById('content');
            const imageInput = document.getElementById('image');
            
            if (contentType === 'text') {
                textContent.style.display = 'block';
                imageContent.style.display = 'none';
                contentTextarea.required = true;
                imageInput.required = false;
            } else {
                textContent.style.display = 'none';
                imageContent.style.display = 'block';
                contentTextarea.required = false;
                imageInput.required = true;
            }
            
            updatePreview();
        });
    });
    
    // Önizleme güncelleme
    function updatePreview() {
        const title = document.getElementById('title').value;
        const content = document.getElementById('content').value;
        const contentType = document.querySelector('input[name="content_type"]:checked').value;
        const backgroundColor = document.getElementById('background_color').value;
        const textColor = document.getElementById('text_color').value;
        const fontSize = document.getElementById('font_size').value;
        
        const preview = document.getElementById('storyPreview');
        preview.style.background = backgroundColor;
        preview.style.color = textColor;
        preview.style.fontSize = fontSize + 'px';
        
        if (contentType === 'text') {
            const previewText = title ? '<strong>' + title + '</strong><br><br>' + content : 'Story önizlemesi burada görünecek';
            preview.innerHTML = '<div>' + previewText + '</div>';
        } else {
            preview.innerHTML = '<div><i class="fas fa-image" style="font-size: 48px; margin-bottom: 10px;"></i><br>Görsel yüklendiğinde<br>burada görünecek</div>';
        }
    }
    
    // Form alanları değiştiğinde önizlemeyi güncelle
    const formInputs = ['title', 'content', 'background_color', 'text_color', 'font_size'];
    formInputs.forEach(function(inputId) {
        const element = document.getElementById(inputId);
        if (element) {
            element.addEventListener('input', updatePreview);
            element.addEventListener('change', updatePreview);
        }
    });
    
    // Sayfa yüklendiğinde önizlemeyi güncelle
    updatePreview();
    
    // Form validasyonu
    const storyForm = document.getElementById('storyForm');
    storyForm.addEventListener('submit', function(e) {
        const contentType = document.querySelector('input[name="content_type"]:checked').value;
        const title = document.getElementById('title').value.trim();
        const content = document.getElementById('content').value.trim();
        const categoryId = document.getElementById('category_id').value;
        const imageFile = document.getElementById('image').files[0];
        
        const errors = [];
        
        if (!title) {
            errors.push('Başlık gereklidir.');
        }
        
        if (!categoryId) {
            errors.push('Kategori seçimi gereklidir.');
        }
        
        if (contentType === 'text' && !content) {
            errors.push('Metin içeriği gereklidir.');
        }
        
        if (contentType === 'image' && !imageFile) {
            errors.push('Görsel dosyası seçmelisiniz.');
        }
        
        if (errors.length > 0) {
            e.preventDefault();
            alert('Lütfen aşağıdaki hataları düzeltin:\n\n' + errors.join('\n'));
            return false;
        }
    });
    
    // Drag & Drop işlevselliği
    const uploadArea = document.querySelector('.file-upload-area');
    
    if (uploadArea) {
        uploadArea.addEventListener('dragover', function(e) {
            e.preventDefault();
            this.classList.add('dragover');
        });
        
        uploadArea.addEventListener('dragleave', function(e) {
            e.preventDefault();
            this.classList.remove('dragover');
        });
        
        uploadArea.addEventListener('drop', function(e) {
            e.preventDefault();
            this.classList.remove('dragover');
            
            const files = e.dataTransfer.files;
            if (files.length > 0) {
                const file = files[0];
                if (file.type.startsWith('image/')) {
                    document.getElementById('image').files = files;
                    previewImage(document.getElementById('image'));
                } else {
                    alert('Lütfen sadece görsel dosyası seçin.');
                }
            }
        });
    }
});

// Görsel önizleme fonksiyonu
function previewImage(input) {
    if (input.files && input.files[0]) {
        var file = input.files[0];
        
        // Dosya boyutu kontrolü (5MB)
        if (file.size > 5 * 1024 * 1024) {
            alert('Dosya boyutu 5MB\'dan küçük olmalıdır.');
            input.value = '';
            return;
        }
        
        // Dosya türü kontrolü
        var allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
        if (!allowedTypes.includes(file.type)) {
            alert('Sadece JPG, PNG, GIF ve WebP formatları desteklenir.');
            input.value = '';
            return;
        }
        
        var reader = new FileReader();
        reader.onload = function(e) {
            var img = new Image();
            img.onload = function() {
                // Boyut kontrolü (1080x1920 ±%10 tolerans)
                var width = this.width;
                var height = this.height;
                
                if (Math.abs(width - 1080) > 108 || Math.abs(height - 1920) > 192) {
                    alert('Görsel boyutu 1080x1920 piksel olmalıdır.\nMevcut boyut: ' + width + 'x' + height + ' piksel');
                    input.value = '';
                    document.getElementById('imagePreviewContainer').style.display = 'none';
                    return;
                }
                
                // Önizlemeyi göster
                document.getElementById('imagePreview').src = e.target.result;
                document.getElementById('imagePreviewContainer').style.display = 'block';
                
                // Story önizlemesini güncelle
                var preview = document.getElementById('storyPreview');
                preview.innerHTML = '<img src="' + e.target.result + '" style="width: 100%; height: 100%; object-fit: cover; border-radius: 10px;">';
            };
            img.src = e.target.result;
        };
        reader.readAsDataURL(file);
    }
}

// Görseli kaldırma fonksiyonu
function removeImage() {
    document.getElementById('image').value = '';
    document.getElementById('imagePreviewContainer').style.display = 'none';
    
    // Story önizlemesini sıfırla
    var preview = document.getElementById('storyPreview');
    preview.innerHTML = '<div><i class="fas fa-image" style="font-size: 48px; margin-bottom: 10px;"></i><br>Görsel yüklendiğinde<br>burada görünecek</div>';
}
</script>

<?php require_once 'includes/footer.php'; ?>
